<?php

/**
 * Фасад для безопасного доступа к суперглобальным переменным
 *
 * @package Flowaxy\Support\Facades
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Facades;

use Flowaxy\Infrastructure\Security\RequestFilter;

final class Request extends Facade
{
    /**
     * Получить имя сервиса в контейнере
     *
     * @return string
     */
    protected static function getFacadeAccessor(): string
    {
        return RequestFilter::class;
    }

    /**
     * Получить экземпляр RequestFilter
     * RequestFilter - статический класс, поэтому возвращаем null
     *
     * @return null
     */
    public static function instance(): ?RequestFilter
    {
        // RequestFilter - статический класс, не требует экземпляра
        return null;
    }

    /**
     * Безопасное получение значения из $_GET
     *
     * @param string $key Ключ параметра
     * @param mixed $default Значение по умолчанию
     * @param string $type Тип данных (string, int, float, email, url, array)
     * @return mixed Отфильтрованное значение
     */
    public static function get(string $key, mixed $default = null, string $type = 'string'): mixed
    {
        return RequestFilter::get($key, $default, $type);
    }

    /**
     * Безопасное получение значения из $_POST
     *
     * @param string $key Ключ параметра
     * @param mixed $default Значение по умолчанию
     * @param string $type Тип данных (string, int, float, email, url, array)
     * @return mixed Отфильтрованное значение
     */
    public static function post(string $key, mixed $default = null, string $type = 'string'): mixed
    {
        return RequestFilter::post($key, $default, $type);
    }

    /**
     * Безопасное получение значения из $_REQUEST
     *
     * @param string $key Ключ параметра
     * @param mixed $default Значение по умолчанию
     * @param string $type Тип данных (string, int, float, email, url, array)
     * @return mixed Отфильтрованное значение
     */
    public static function request(string $key, mixed $default = null, string $type = 'string'): mixed
    {
        return RequestFilter::request($key, $default, $type);
    }

    /**
     * Безопасное получение значения из $_SERVER
     *
     * @param string $key Ключ параметра
     * @param mixed $default Значение по умолчанию
     * @param string $type Тип данных (string, int, float, email, url, array)
     * @return mixed Отфильтрованное значение
     */
    public static function server(string $key, mixed $default = null, string $type = 'string'): mixed
    {
        return RequestFilter::server($key, $default, $type);
    }

    /**
     * Безопасное получение значения из $_COOKIE
     *
     * @param string $key Ключ параметра
     * @param mixed $default Значение по умолчанию
     * @param string $type Тип данных (string, int, float, email, url, array)
     * @return mixed Отфильтрованное значение
     */
    public static function cookie(string $key, mixed $default = null, string $type = 'string'): mixed
    {
        return RequestFilter::cookie($key, $default, $type);
    }

    /**
     * Получить все отфильтрованные GET параметры
     *
     * @return array<string, mixed>
     */
    public static function allGet(): array
    {
        return RequestFilter::allGet();
    }

    /**
     * Получить все отфильтрованные POST параметры
     *
     * @return array<string, mixed>
     */
    public static function allPost(): array
    {
        return RequestFilter::allPost();
    }

    /**
     * Получить все отфильтрованные SERVER переменные
     *
     * @return array<string, mixed>
     */
    public static function allServer(): array
    {
        return RequestFilter::allServer();
    }

    /**
     * Очистить кеш фильтра
     *
     * @return void
     */
    public static function clearCache(): void
    {
        RequestFilter::clearCache();
    }
}
