<?php

/**
 * Фасад для роботи з JSON
 *
 * @package Flowaxy\Support\Facades
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Facades;

use Flowaxy\Infrastructure\Filesystem\Json as JsonInstance;
use Flowaxy\Support\Helpers\JsonHelper;

final class Json extends Facade
{
    protected static function getFacadeAccessor(): string
    {
        return JsonInstance::class;
    }

    /**
     * Кодувати дані в JSON
     *
     * @param mixed $value
     * @param int $flags
     * @return string|false
     */
    public static function encode(mixed $value, int $flags = JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES): string|false
    {
        return JsonHelper::encode($value, $flags);
    }

    /**
     * Декодувати JSON в дані
     *
     * @param string $json
     * @param bool $assoc
     * @return mixed
     */
    public static function decode(string $json, bool $assoc = true): mixed
    {
        return JsonHelper::decode($json, $assoc);
    }

    /**
     * Кодувати дані в JSON з форматуванням
     *
     * @param mixed $value
     * @param int $flags
     * @return string|false
     */
    public static function encodePretty(mixed $value, int $flags = JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES): string|false
    {
        return JsonHelper::encodePretty($value, $flags);
    }

    /**
     * Перевірити, чи рядок є валідним JSON
     *
     * @param string $json
     * @return bool
     */
    public static function isValid(string $json): bool
    {
        return JsonHelper::isValid($json);
    }

    /**
     * Прочитати JSON з файлу
     *
     * @param string $path
     * @param bool $assoc
     * @return mixed|false
     */
    public static function readFile(string $path, bool $assoc = true): mixed
    {
        return JsonHelper::readFile($path, $assoc);
    }

    /**
     * Записати JSON у файл
     *
     * @param string $path
     * @param mixed $data
     * @param bool $pretty
     * @return bool
     */
    public static function writeFile(string $path, mixed $data, bool $pretty = false): bool
    {
        return JsonHelper::writeFile($path, $data, $pretty);
    }

    /**
     * Об'єднати JSON дані
     *
     * @param mixed ...$data
     * @return array<string, mixed>|false
     */
    public static function merge(mixed ...$data): array|false
    {
        return JsonHelper::merge(...$data);
    }
}
