<?php

/**
 * Фасад для роботи з INI
 *
 * @package Flowaxy\Support\Facades
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Facades;

use Flowaxy\Infrastructure\Filesystem\Ini as IniInstance;
use Flowaxy\Support\Helpers\IniHelper;

final class Ini extends Facade
{
    protected static function getFacadeAccessor(): string
    {
        return IniInstance::class;
    }

    /**
     * Парсити INI рядок
     *
     * @param string $ini
     * @param bool $processSections
     * @return array<string, mixed>|false
     */
    public static function parse(string $ini, bool $processSections = true): array|false
    {
        return IniHelper::parse($ini, $processSections);
    }

    /**
     * Генерувати INI з даних
     *
     * @param array<string, mixed> $data
     * @return string
     */
    public static function dump(array $data): string
    {
        return IniHelper::dump($data);
    }

    /**
     * Прочитати INI з файлу
     *
     * @param string $path
     * @param bool $processSections
     * @return array<string, mixed>|false
     */
    public static function readFile(string $path, bool $processSections = true): array|false
    {
        return IniHelper::readFile($path, $processSections);
    }

    /**
     * Записати INI у файл
     *
     * @param string $path
     * @param array<string, mixed> $data
     * @return bool
     */
    public static function writeFile(string $path, array $data): bool
    {
        return IniHelper::writeFile($path, $data);
    }

    /**
     * Отримати секцію з INI
     *
     * @param array<string, mixed> $ini
     * @param string $section
     * @return array<string, mixed>|false
     */
    public static function getSection(array $ini, string $section): array|false
    {
        return IniHelper::getSection($ini, $section);
    }

    /**
     * Встановити секцію в INI
     *
     * @param array<string, mixed> $ini
     * @param string $section
     * @param array<string, mixed> $values
     * @return array<string, mixed>
     */
    public static function setSection(array &$ini, string $section, array $values): array
    {
        return IniHelper::setSection($ini, $section, $values);
    }

    /**
     * Отримати значення з INI
     *
     * @param array<string, mixed> $ini
     * @param string $key
     * @param string|null $section
     * @param mixed $default
     * @return mixed
     */
    public static function getValue(array $ini, string $key, ?string $section = null, mixed $default = null): mixed
    {
        return IniHelper::getValue($ini, $key, $section, $default);
    }

    /**
     * Встановити значення в INI
     *
     * @param array<string, mixed> $ini
     * @param string $key
     * @param mixed $value
     * @param string|null $section
     * @return array<string, mixed>
     */
    public static function setValue(array &$ini, string $key, mixed $value, ?string $section = null): array
    {
        return IniHelper::setValue($ini, $key, $value, $section);
    }
}
