<?php

/**
 * Фасад для роботи з зображеннями
 *
 * @package Flowaxy\Support\Facades
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Facades;

use Flowaxy\Infrastructure\Filesystem\Image as ImageInstance;
use Flowaxy\Support\Helpers\ImageHelper;

final class Image extends Facade
{
    protected static function getFacadeAccessor(): string
    {
        return ImageInstance::class;
    }

    /**
     * Створити екземпляр Image
     *
     * @param string|null $filePath
     * @return ImageInstance
     */
    public static function make(?string $filePath = null): ImageInstance
    {
        return new ImageInstance($filePath);
    }

    /**
     * Змінити розмір зображення
     *
     * @param string $filePath
     * @param int $width
     * @param int $height
     * @param bool $maintainAspectRatio
     * @return ImageInstance
     */
    public static function resize(string $filePath, int $width, int $height, bool $maintainAspectRatio = true): ImageInstance
    {
        $image = new ImageInstance($filePath);
        $image->resize($width, $height, $maintainAspectRatio);

        return $image;
    }

    /**
     * Обрізати зображення
     *
     * @param string $filePath
     * @param int $x
     * @param int $y
     * @param int $width
     * @param int $height
     * @return ImageInstance
     */
    public static function crop(string $filePath, int $x, int $y, int $width, int $height): ImageInstance
    {
        $image = new ImageInstance($filePath);
        $image->crop($x, $y, $width, $height);

        return $image;
    }

    /**
     * Повернути зображення
     *
     * @param string $filePath
     * @param int $angle
     * @return ImageInstance
     */
    public static function rotate(string $filePath, int $angle): ImageInstance
    {
        $image = new ImageInstance($filePath);
        $image->rotate($angle);

        return $image;
    }

    /**
     * Відобразити зображення дзеркально
     *
     * @param string $filePath
     * @param string $mode
     * @return ImageInstance
     */
    public static function flip(string $filePath, string $mode = 'horizontal'): ImageInstance
    {
        $image = new ImageInstance($filePath);
        $image->flip($mode);

        return $image;
    }

    /**
     * Додати водяний знак
     *
     * @param string $filePath
     * @param string $watermarkPath
     * @param string $position
     * @param int $opacity
     * @return ImageInstance
     */
    public static function watermark(string $filePath, string $watermarkPath, string $position = 'bottom-right', int $opacity = 50): ImageInstance
    {
        $image = new ImageInstance($filePath);
        $image->watermark($watermarkPath, $position, $opacity);

        return $image;
    }

    /**
     * Зберегти зображення
     *
     * @param string $filePath
     * @param string $savePath
     * @param int $quality
     * @return bool
     */
    public static function save(string $filePath, string $savePath, int $quality = 90): bool
    {
        $image = new ImageInstance($filePath);

        return $image->save($savePath, $quality);
    }

    /**
     * Вивести зображення
     *
     * @param string $filePath
     * @param int $quality
     * @return void
     */
    public static function output(string $filePath, int $quality = 90): void
    {
        $image = new ImageInstance($filePath);
        $image->output($quality);
    }

    /**
     * Отримати base64 рядок зображення
     *
     * @param string $filePath
     * @return string|false
     */
    public static function base64(string $filePath): string|false
    {
        return ImageHelper::base64($filePath);
    }

    /**
     * Створити мініатюру
     *
     * @param string $filePath
     * @param int $width
     * @param int $height
     * @param string $savePath
     * @return bool
     */
    public static function thumbnail(string $filePath, int $width, int $height, string $savePath): bool
    {
        return ImageHelper::thumbnail($filePath, $width, $height, $savePath);
    }

    /**
     * Підігнати зображення під розміри
     *
     * @param string $filePath
     * @param int $width
     * @param int $height
     * @param string $savePath
     * @return bool
     */
    public static function fit(string $filePath, int $width, int $height, string $savePath): bool
    {
        return ImageHelper::fit($filePath, $width, $height, $savePath);
    }
}
