<?php

/**
 * Фасад для роботи з хешуванням
 *
 * @package Flowaxy\Support\Facades
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Facades;

use Flowaxy\Infrastructure\Security\Hash as HashInstance;
use Flowaxy\Support\Facades\Log;

use function password_get_info;

final class Hash extends Facade
{
    protected static function getFacadeAccessor(): string
    {
        return HashInstance::class;
    }

    /**
     * Отримати екземпляр Hash
     *
     * @return HashInstance
     */
    private static function hasher(): HashInstance
    {
        try {
            return static::getFacadeRoot();
        } catch (\RuntimeException $e) {
            // Если HashInstance не найден в контейнере, логируем ошибку и создаем напрямую
            // Это fallback для случаев, когда контейнер еще не инициализирован или HashInstance не зарегистрирован
            try {
                Log::Error('Hash facade: HashInstance not found in container, using direct instantiation', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            } catch (\Throwable $logError) {
                // Игнорируем ошибки логирования
            }

            // Fallback: создаем HashInstance напрямую
            return new HashInstance();
        }
    }

    /**
     * Створити хеш
     *
     * @param string $password
     * @param int|null $algorithm
     * @param array<string, mixed>|null $options
     * @return string
     */
    public static function make(string $password, ?int $algorithm = null, ?array $options = null): string
    {
        $hasher = static::hasher();
        $optionsArray = [];
        if ($algorithm !== null) {
            $optionsArray['algorithm'] = $algorithm;
        }
        if ($options !== null) {
            $optionsArray['options'] = $options;
        }
        return $hasher->make($password, $optionsArray);
    }

    /**
     * Перевірити хеш
     *
     * @param string $password
     * @param string $hash
     * @return bool
     */
    public static function check(string $password, string $hash): bool
    {
        return static::hasher()->check($password, $hash);
    }

    /**
     * Перевірити, чи потрібно перехешувати
     *
     * @param string $hash
     * @param int|null $algorithm
     * @param array<string, mixed>|null $options
     * @return bool
     */
    public static function needsRehash(string $hash, ?int $algorithm = null, ?array $options = null): bool
    {
        $hasher = static::hasher();
        $optionsArray = [];
        if ($algorithm !== null) {
            $optionsArray['algorithm'] = $algorithm;
        }
        if ($options !== null) {
            $optionsArray['options'] = $options;
        }
        return $hasher->needsRehash($hash, $optionsArray);
    }

    /**
     * Отримати інформацію про хеш
     *
     * @param string $hash
     * @return array<string, mixed>
     */
    public static function info(string $hash): array
    {
        return static::hasher()->info($hash);
    }

    /**
     * Хешувати значення (alias для make)
     *
     * @param string $password
     * @param int|null $algorithm
     * @param array<string, mixed>|null $options
     * @return string
     */
    public static function hash(string $password, ?int $algorithm = null, ?array $options = null): string
    {
        return static::make($password, $algorithm, $options);
    }

    /**
     * Перевірити хеш (alias для check)
     *
     * @param string $value
     * @param string $hash
     * @return bool
     */
    public static function verify(string $value, string $hash): bool
    {
        return static::check($value, $hash);
    }
}
