<?php

/**
 * Фасад для роботи з CSV
 *
 * @package Flowaxy\Support\Facades
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Facades;

use Flowaxy\Infrastructure\Filesystem\Csv as CsvInstance;
use Flowaxy\Support\Helpers\CsvHelper;

final class Csv extends Facade
{
    protected static function getFacadeAccessor(): string
    {
        return CsvInstance::class;
    }

    /**
     * Парсити CSV рядок
     *
     * @param string $csv
     * @param string $delimiter
     * @return array<int, array<int, string>>
     */
    public static function parse(string $csv, string $delimiter = ','): array
    {
        return CsvHelper::parse($csv, $delimiter);
    }

    /**
     * Конвертувати CSV в масив
     *
     * @param string $csv
     * @param bool $hasHeaders
     * @param string $delimiter
     * @return array<int, array<string, mixed>>
     */
    public static function toArray(string $csv, bool $hasHeaders = true, string $delimiter = ','): array
    {
        return CsvHelper::toArray($csv, $hasHeaders, $delimiter);
    }

    /**
     * Конвертувати масив в CSV
     *
     * @param array<int, array<string|int, mixed>> $array
     * @param array<int, string>|null $headers
     * @param string $delimiter
     * @return string
     */
    public static function fromArray(array $array, ?array $headers = null, string $delimiter = ','): string
    {
        return CsvHelper::fromArray($array, $headers, $delimiter);
    }

    /**
     * Прочитати CSV з файлу
     *
     * @param string $path
     * @param bool $hasHeaders
     * @param string $delimiter
     * @return array<int, array<string, mixed>>
     */
    public static function readFile(string $path, bool $hasHeaders = true, string $delimiter = ','): array
    {
        return CsvHelper::readFile($path, $hasHeaders, $delimiter);
    }

    /**
     * Записати CSV у файл
     *
     * @param string $path
     * @param array<int, array<string|int, mixed>> $array
     * @param array<int, string>|null $headers
     * @param string $delimiter
     * @return bool
     */
    public static function writeFile(string $path, array $array, ?array $headers = null, string $delimiter = ','): bool
    {
        return CsvHelper::writeFile($path, $array, $headers, $delimiter);
    }

    /**
     * Отримати заголовки з CSV
     *
     * @param string $csv
     * @param string $delimiter
     * @return array<int, string>|false
     */
    public static function getHeaders(string $csv, string $delimiter = ','): array|false
    {
        return CsvHelper::getHeaders($csv, $delimiter);
    }

    /**
     * Отримати рядок з CSV
     *
     * @param string $csv
     * @param int $index
     * @param bool $hasHeaders
     * @param string $delimiter
     * @return array<string|int, mixed>|false
     */
    public static function getRow(string $csv, int $index, bool $hasHeaders = true, string $delimiter = ','): array|false
    {
        return CsvHelper::getRow($csv, $index, $hasHeaders, $delimiter);
    }

    /**
     * Додати рядок до CSV
     *
     * @param string $csv
     * @param array<string|int, mixed> $row
     * @param string $delimiter
     * @return string
     */
    public static function addRow(string $csv, array $row, string $delimiter = ','): string
    {
        return CsvHelper::addRow($csv, $row, $delimiter);
    }
}
