<?php

/**
 * Фасад для роботи з конфігурацією
 *
 * @package Flowaxy\Support\Facades
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Facades;

use Flowaxy\Infrastructure\Config\SystemConfig;
use Flowaxy\Support\Helpers\ConfigHelper;
use Flowaxy\Support\Facades\Settings as SettingsFacade;

use function class_exists;

final class Config extends Facade
{
    protected static function getFacadeAccessor(): string
    {
        return SystemConfig::class;
    }

    /**
     * Отримати значення конфігурації
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public static function get(string $key, mixed $default = null): mixed
    {
        return SystemConfig::getInstance()->get($key, $default);
    }

    /**
     * Встановити значення конфігурації
     *
     * @param string $key
     * @param mixed $value
     * @return void
     */
    public static function set(string $key, mixed $value): void
    {
        if (class_exists(SettingsFacade::class)) {
            SettingsFacade::set($key, $value);
        }
    }

    /**
     * Перевірити наявність ключа
     *
     * @param string $key
     * @return bool
     */
    public static function has(string $key): bool
    {
        if (class_exists(SettingsFacade::class)) {
            return SettingsFacade::has($key);
        }

        return self::get($key) !== null;
    }

    /**
     * Отримати всі значення конфігурації
     *
     * @return array<string, mixed>
     */
    public static function all(): array
    {
        if (class_exists(SettingsFacade::class)) {
            $manager = SettingsFacade::manager();
            if ($manager !== null) {
                return $manager->all();
            }
        }

        return [];
    }

    /**
     * Завантажити конфігурацію
     *
     * @param string $path
     * @return array<string, mixed>|false
     */
    public static function load(string $path): array|false
    {
        if (class_exists(ConfigHelper::class)) {
            return ConfigHelper::load($path);
        }

        return false;
    }

    /**
     * Зберегти конфігурацію
     *
     * @param string $path
     * @param array<string, mixed> $data
     * @return bool
     */
    public static function save(string $path, array $data): bool
    {
        if (class_exists(ConfigHelper::class)) {
            return ConfigHelper::save($path, $data);
        }

        return false;
    }
}
