<?php

/**
 * Фасад для роботи з ассетами
 *
 * @package Flowaxy\Support\Facades
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Facades;

use Flowaxy\Infrastructure\Assets\AssetManager;
use Flowaxy\Support\Helpers\UrlHelper;
use Flowaxy\Support\Helpers\AssetHelper;

final class Asset extends Facade
{
    protected static function getFacadeAccessor(): string
    {
        return AssetManager::class;
    }

    /**
     * Отримати менеджер ассетів
     *
     * @return AssetManager
     */
    public static function manager(): AssetManager
    {
        static $manager = null;

        if ($manager === null) {
            $manager = new AssetManager();
        }

        return $manager;
    }

    /**
     * Додати CSS файл
     *
     * @param string $path
     * @param string|null $key
     * @return void
     */
    public static function css(string $path, ?string $key = null): void
    {
        self::manager()->addCss($path, $key);
    }

    /**
     * Додати JS файл
     *
     * @param string $path
     * @param string|null $key
     * @return void
     */
    public static function js(string $path, ?string $key = null): void
    {
        self::manager()->addJs($path, $key);
    }

    /**
     * Отримати URL зображення
     *
     * @param string $path
     * @return string
     */
    public static function image(string $path): string
    {
        return UrlHelper::uploads('images/' . ltrim($path, '/'));
    }

    /**
     * Отримати URL ассета
     *
     * @param string $path
     * @return string
     */
    public static function url(string $path): string
    {
        return UrlHelper::admin('assets/' . ltrim($path, '/'));
    }

    /**
     * Отримати URL з версією
     *
     * @param string $path
     * @return string
     */
    public static function version(string $path): string
    {
        return AssetHelper::version($path);
    }

    /**
     * Об'єднати ассети
     *
     * @param array<int, string> $paths
     * @param string $output
     * @return bool
     */
    public static function combine(array $paths, string $output): bool
    {
        return AssetHelper::combine($paths, $output);
    }

    /**
     * Мініфікувати ассет
     *
     * @param string $path
     * @param string|null $output
     * @return bool
     */
    public static function minify(string $path, ?string $output = null): bool
    {
        return AssetHelper::minify($path, $output);
    }

    /**
     * Вставити ассет inline
     *
     * @param string $path
     * @return string
     */
    public static function inline(string $path): string
    {
        return AssetHelper::inline($path);
    }

    /**
     * Додати preload для ассета
     *
     * @param string $path
     * @param string $as
     * @return string
     */
    public static function preload(string $path, string $as = 'style'): string
    {
        return AssetHelper::preload($path, $as);
    }
}
