<?php

/**
 * Фасад для роботи з архівами
 *
 * @package Flowaxy\Support\Facades
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Facades;

use Flowaxy\Infrastructure\Filesystem\Archive\Zip;
use Flowaxy\Support\Helpers\ArchiveHelper;

final class Archive extends Facade
{
    protected static function getFacadeAccessor(): string
    {
        return Zip::class;
    }

    /**
     * Розпакувати архів
     *
     * @param string $archivePath
     * @param string $destinationPath
     * @return bool
     */
    public static function extract(string $archivePath, string $destinationPath): bool
    {
        return ArchiveHelper::extract($archivePath, $destinationPath);
    }

    /**
     * Створити архів
     *
     * @param string|array<int, string> $paths
     * @param string $archivePath
     * @return bool
     */
    public static function compress(string|array $paths, string $archivePath): bool
    {
        return \Flowaxy\Support\Helpers\ArchiveHelper::compress($paths, $archivePath);
    }

    /**
     * Додати файл до архіву
     *
     * @param string $archivePath
     * @param string $filePath
     * @param string|null $localName
     * @return bool
     */
    public static function addFile(string $archivePath, string $filePath, ?string $localName = null): bool
    {
        return \Flowaxy\Support\Helpers\ArchiveHelper::addFile($archivePath, $filePath, $localName);
    }

    /**
     * Додати директорію до архіву
     *
     * @param string $archivePath
     * @param string $directoryPath
     * @return bool
     */
    public static function addDirectory(string $archivePath, string $directoryPath): bool
    {
        return \Flowaxy\Support\Helpers\ArchiveHelper::addDirectory($archivePath, $directoryPath);
    }

    /**
     * Отримати список файлів в архіві
     *
     * @param string $archivePath
     * @return array<int, string>|false
     */
    public static function listFiles(string $archivePath): array|false
    {
        return \Flowaxy\Support\Helpers\ArchiveHelper::listFiles($archivePath);
    }

    /**
     * Отримати файл з архіву
     *
     * @param string $archivePath
     * @param string $fileName
     * @return string|false
     */
    public static function getFile(string $archivePath, string $fileName): string|false
    {
        return \Flowaxy\Support\Helpers\ArchiveHelper::getFile($archivePath, $fileName);
    }

    /**
     * Видалити файл з архіву
     *
     * @param string $archivePath
     * @param string $fileName
     * @return bool
     */
    public static function deleteFile(string $archivePath, string $fileName): bool
    {
        return \Flowaxy\Support\Helpers\ArchiveHelper::deleteFile($archivePath, $fileName);
    }

    /**
     * Перевірити, чи архів валідний
     *
     * @param string $archivePath
     * @return bool
     */
    public static function isValid(string $archivePath): bool
    {
        return \Flowaxy\Support\Helpers\ArchiveHelper::isValid($archivePath);
    }

    /**
     * Відкрити архів
     *
     * @param string $filePath
     * @param int $flags
     * @return Zip
     */
    public static function open(string $filePath, int $flags = 1): Zip
    {
        $zip = new Zip();
        $zip->open($filePath, $flags);
        return $zip;
    }
}
