<?php

declare(strict_types=1);

namespace Flowaxy\Support\Actions\Theme;

use Flowaxy\Support\Actions\ActionInterface;
use Flowaxy\Support\Services\ThemeService;
use Flowaxy\Support\Helpers\SessionHelper;
use Flowaxy\Support\Facades\Log;
use Flowaxy\Support\Managers\ThemeManager;
use Flowaxy\Support\Helpers\ManagerHelper;
use Exception;
use Throwable;

use function count;
use function preg_match;

// Действие удаления темы
final class DeleteThemeAction implements ActionInterface
{
    private ThemeService $themeService;

    public function __construct()
    {
        $this->themeService = new ThemeService();
    }

    public function execute(array $params = []): array
    {
        $themeSlug = $params['theme_slug'] ?? '';

        if (empty($themeSlug)) {
            return [
                'success' => false,
                'error' => 'Slug теми не вказано',
            ];
        }

        try {
            // Проверяем, что тема не активна
            $themeManager = ManagerHelper::themeManager();
            if ($themeManager) {
                $activeTheme = $themeManager->getActiveTheme();
                if ($activeTheme && isset($activeTheme['slug']) && $activeTheme['slug'] === $themeSlug) {
                    return [
                        'success' => false,
                        'error' => 'Неможливо видалити активну тему. Спочатку активуйте іншу тему.',
                    ];
                }

                // Проверяем, является ли тема стандартной
                $theme = $themeManager->getTheme($themeSlug);
                $isDefault = $theme && ($theme['is_default'] ?? false);

                if ($isDefault) {
                    $allThemes = $themeManager->getAllThemes();
                    $themesCount = count($allThemes);
                    $userId = SessionHelper::getUserId();

                    // Если тема одна и стандартная - только разработчик может её удалить
                    if ($themesCount === 1) {
                        if ($userId !== 1) {
                            return [
                                'success' => false,
                                'error' => 'Неможливо видалити стандартну встановлену тему. Це єдина доступна тема.',
                            ];
                        }
                    } else {
                        // Если тем больше одной, стандартную тему нельзя удалять
                        return [
                            'success' => false,
                            'error' => 'Неможливо видалити стандартну встановлену тему.',
                        ];
                    }
                }
            }

            $result = $this->themeService->delete($themeSlug);

            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Тему видалено',
                    'data' => ['theme_slug' => $themeSlug],
                ];
            }

            return [
                'success' => false,
                'error' => 'Не вдалося видалити тему',
            ];
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка видалення теми', ['exception' => $e, 'theme_slug' => $themeSlug]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function validate(array $params): array
    {
        if (empty($params['theme_slug'])) {
            return [
                'valid' => false,
                'error' => 'Slug теми обов\'язковий',
            ];
        }

        $themeSlug = (string)$params['theme_slug'];
        if (!preg_match('/^[a-z0-9\-_]+$/i', $themeSlug)) {
            return [
                'valid' => false,
                'error' => 'Невірний формат slug теми',
            ];
        }

        return ['valid' => true];
    }

    public function getName(): string
    {
        return 'delete_theme';
    }

    public function getDescription(): string
    {
        return 'Видалення теми';
    }
}
