<?php

declare(strict_types=1);

namespace Flowaxy\Support\Actions\Plugin;

use Flowaxy\Support\Actions\ActionInterface;
use Flowaxy\Support\Services\PluginService;
use Flowaxy\Support\Helpers\DatabaseHelper;
use Flowaxy\Support\Facades\Log;
use Exception;
use Throwable;
use PDO;

// Действие удаления плагина
final class UninstallPluginAction implements ActionInterface
{
    private PluginService $pluginService;

    public function __construct()
    {
        $this->pluginService = new PluginService();
    }

    public function execute(array $params = []): array
    {
        $pluginSlug = $params['plugin_slug'] ?? '';

        if (empty($pluginSlug)) {
            return [
                'success' => false,
                'error' => 'Slug плагіна не вказано',
            ];
        }

        try {
            // Проверяем, что плагин деактивирован
            $db = DatabaseHelper::getConnection();
            if ($db) {
                $stmt = $db->prepare('SELECT is_active FROM plugins WHERE slug = ?');
                $stmt->execute([$pluginSlug]);
                $plugin = $stmt->fetch(PDO::FETCH_ASSOC);

                if ($plugin && !empty($plugin['is_active']) && $plugin['is_active'] == 1) {
                    return [
                        'success' => false,
                        'error' => 'Спочатку деактивуйте плагін перед видаленням',
                    ];
                }
            }

            $result = $this->pluginService->uninstall($pluginSlug);

            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Плагін видалено',
                    'data' => ['plugin_slug' => $pluginSlug],
                ];
            }

            return [
                'success' => false,
                'error' => 'Не вдалося видалити плагін',
            ];
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка видалення плагіна', ['exception' => $e, 'plugin_slug' => $pluginSlug]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function validate(array $params): array
    {
        if (empty($params['plugin_slug'])) {
            return [
                'valid' => false,
                'error' => 'Slug плагіна обов\'язковий',
            ];
        }

        return ['valid' => true];
    }

    public function getName(): string
    {
        return 'uninstall_plugin';
    }

    public function getDescription(): string
    {
        return 'Видалення плагіна';
    }
}
