<?php

/**
 * Действие установки плагина
 *
 * @package Flowaxy\Support\Actions\Plugin
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Actions\Plugin;

use Flowaxy\Support\Actions\ActionInterface;
use Flowaxy\Support\Services\PluginService;
use Flowaxy\Support\Facades\Log;
use Exception;
use Throwable;

final class InstallPluginAction implements ActionInterface
{
    private PluginService $pluginService;

    public function __construct()
    {
        $this->pluginService = new PluginService();
    }

    /**
     * Выполнение действия
     *
     * @param array<string, mixed> $params Параметры действия
     * @return array{success: bool, message?: string, error?: string, data?: mixed}
     */
    public function execute(array $params = []): array
    {
        $pluginSlug = $params['plugin_slug'] ?? '';

        if (empty($pluginSlug)) {
            return [
                'success' => false,
                'error' => 'Slug плагіна не вказано',
            ];
        }

        try {
            $result = $this->pluginService->install($pluginSlug);

            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Плагін встановлено',
                    'data' => ['plugin_slug' => $pluginSlug],
                ];
            }

            return [
                'success' => false,
                'error' => 'Не вдалося встановити плагін',
            ];
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка встановлення плагіна', ['exception' => $e, 'plugin_slug' => $pluginSlug]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Валидация параметров
     *
     * @param array<string, mixed> $params Параметры действия
     * @return array{valid: bool, error?: string}
     */
    public function validate(array $params): array
    {
        if (empty($params['plugin_slug'])) {
            return [
                'valid' => false,
                'error' => 'Slug плагіна обов\'язковий',
            ];
        }

        $pluginSlug = (string)$params['plugin_slug'];
        if (!preg_match('/^[a-z0-9\-_]+$/i', $pluginSlug)) {
            return [
                'valid' => false,
                'error' => 'Невірний формат slug плагіна',
            ];
        }

        return ['valid' => true];
    }

    /**
     * Получение имени действия
     *
     * @return string
     */
    public function getName(): string
    {
        return 'install_plugin';
    }

    /**
     * Получение описания действия
     *
     * @return string
     */
    public function getDescription(): string
    {
        return 'Встановлення плагіна';
    }
}
