<?php

/**
 * Действие активации плагина
 *
 * @package Flowaxy\Support\Actions\Plugin
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Actions\Plugin;

use Flowaxy\Support\Actions\ActionInterface;
use Flowaxy\Support\Services\PluginService;
use Flowaxy\Support\Facades\Log;
use Exception;
use Throwable;

final class ActivatePluginAction implements ActionInterface
{
    private PluginService $pluginService;

    public function __construct()
    {
        $this->pluginService = new PluginService();
    }

    public function execute(array $params = []): array
    {
        $pluginSlug = $params['plugin_slug'] ?? '';

        if (empty($pluginSlug)) {
            return [
                'success' => false,
                'error' => 'Slug плагіна не вказано',
            ];
        }

        try {
            $autoInstallDependencies = (bool)($params['auto_install_dependencies'] ?? false);
            $result = $this->pluginService->activate($pluginSlug, $autoInstallDependencies);

            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Плагін активовано',
                    'data' => ['plugin_slug' => $pluginSlug],
                ];
            }

            return [
                'success' => false,
                'error' => 'Не вдалося активувати плагін',
            ];
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка активації плагіна', ['exception' => $e, 'plugin_slug' => $pluginSlug]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function validate(array $params): array
    {
        if (empty($params['plugin_slug'])) {
            return [
                'valid' => false,
                'error' => 'Slug плагіна обов\'язковий',
            ];
        }

        $pluginSlug = (string)$params['plugin_slug'];
        if (!preg_match('/^[a-z0-9\-_]+$/i', $pluginSlug)) {
            return [
                'valid' => false,
                'error' => 'Невірний формат slug плагіна',
            ];
        }

        return ['valid' => true];
    }

    public function getName(): string
    {
        return 'activate_plugin';
    }

    public function getDescription(): string
    {
        return 'Активація плагіна';
    }
}
