<?php

/**
 * Базовий клас для HTTP контролерів
 *
 * @package Flowaxy\Interface\Http
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Interface\Http;

use Flowaxy\Interface\Http\Request;
use Flowaxy\Interface\Http\Response;

abstract class Controller
{
    protected Request $request;

    public function __construct()
    {
        $this->request = Request::getInstance();
    }

    /**
     * Викликається перед виконанням методу
     *
     * @return void
     */
    protected function beforeAction(): void
    {
        // Перевизначається в дочірніх класах
    }

    /**
     * Викликається після виконання методу
     *
     * @param mixed $result Результат виконання методу
     * @return mixed
     */
    protected function afterAction($result)
    {
        return $result;
    }

    /**
     * Створити успішну відповідь
     *
     * @param mixed $data Дані для відповіді
     * @param int $statusCode HTTP статус код
     * @return Response
     */
    protected function success($data = null, int $statusCode = 200): Response
    {
        return new Response($data, $statusCode);
    }

    /**
     * Створити помилкову відповідь
     *
     * @param string $message Повідомлення про помилку
     * @param int $statusCode HTTP статус код
     * @return Response
     */
    protected function error(string $message, int $statusCode = 400): Response
    {
        return new Response(['error' => $message], $statusCode);
    }

    /**
     * Створити JSON відповідь
     *
     * @param mixed $data Дані для відповіді
     * @param int $statusCode HTTP статус код
     * @return Response
     */
    protected function json($data, int $statusCode = 200): Response
    {
        $response = new Response($data, $statusCode);
        $response->header('Content-Type', 'application/json');
        return $response;
    }
}
