<?php

/**
 * Middleware для аутентифікації API
 *
 * @package Flowaxy\Interface\API\Middleware
 * @version 1.0.0
 */

declare(strict_types=1);

namespace Flowaxy\Interface\API\Middleware;

use Flowaxy\Interface\API\ApiResponse;
use Flowaxy\Support\Facades\SessionFacade;

final class AuthMiddleware
{
    /**
     * Перевірка аутентифікації через токен
     *
     * @return callable
     */
    public static function token(): callable
    {
        return function () {
            $token = self::getTokenFromRequest();

            if (empty($token)) {
                ApiResponse::error('Unauthorized', 401)->send();
                return;
            }

            if (!self::validateToken($token)) {
                ApiResponse::error('Invalid token', 401)->send();
                return;
            }

            // Токен валідний, продовжуємо
            return null;
        };
    }

    /**
     * Отримання токена з запиту
     */
    private static function getTokenFromRequest(): string
    {
        // Перевіряємо заголовок Authorization
        $headers = \getallheaders();
        $authHeader = $headers['Authorization'] ?? $headers['authorization'] ?? '';

        if (preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
            return $matches[1];
        }

        // Перевіряємо параметр запиту
        return \Flowaxy\Infrastructure\Security\RequestFilter::get('token',
            \Flowaxy\Infrastructure\Security\RequestFilter::post('token', '', 'string'),
            'string'
        );
    }

    /**
     * Валідація токена
     */
    private static function validateToken(string $token): bool
    {
        // Спрощена реалізація - в реальності потрібна перевірка в БД
        try {
            $session = SessionFacade::manager();
            $storedToken = $session->get('api_token');

            return $storedToken === $token;
        } catch (\Throwable $e) {
            return false;
        }
    }
}
