<?php

declare(strict_types=1);

namespace Flowaxy\Infrastructure\Security\Validators;

use Flowaxy\Core\System\PathResolver;
use Flowaxy\Support\Helpers\ClassLoaderHelper;
use Flowaxy\Support\Helpers\SanitizationHelper;
use Throwable;

use const DS;

// Санитизатор входных данных
final class Sanitizer
{
    // Убеждаемся, что SanitizationHelper загружен
    private static function ensureSanitizationHelper(): void
    {
        if (class_exists(SanitizationHelper::class, false)) {
            return;
        }

        // Пробуем использовать ClassLoaderHelper, если он доступен
        if (class_exists(ClassLoaderHelper::class, false)) {
            ClassLoaderHelper::ensureLoaded(SanitizationHelper::class);
            return;
        }

        // Fallback: загружаем напрямую через PathResolver
        if (class_exists(PathResolver::class)) {
            try {
                $flowaxyDir = PathResolver::flowaxy();
                $helperFile = $flowaxyDir . DS . 'Support' . DS . 'Helpers' . DS . 'SanitizationHelper.php';
                if (file_exists($helperFile)) {
                    require_once $helperFile;
                    return;
                }
            } catch (Throwable $e) {
                // PathResolver не доступен, пробуем fallback
            }
        }

        // Fallback: относительный путь от текущего файла
        $currentDir = __DIR__; // flowaxy/Infrastructure/Security/Validators/
        $flowaxyDir = dirname(dirname(dirname(dirname($currentDir)))); // flowaxy/
        $helperFile = $flowaxyDir . DS . 'Support' . DS . 'Helpers' . DS . 'SanitizationHelper.php';
        if (file_exists($helperFile)) {
            require_once $helperFile;
        }
    }

    // Очистка строки от опасных символов
    public function sanitizeString(string $input): string
    {
        self::ensureSanitizationHelper();

        // Используем SanitizationHelper как единую точку входа
        return SanitizationHelper::sanitizeString($input, true);
    }

    // Очистка массива
    public function sanitizeArray(array $input): array
    {
        self::ensureSanitizationHelper();

        // Используем SanitizationHelper как единую точку входа
        return SanitizationHelper::sanitizeArray($input);
    }

    // Очистка email
    public function sanitizeEmail(string $email): string
    {
        self::ensureSanitizationHelper();

        // Используем SanitizationHelper как единую точку входа
        return SanitizationHelper::sanitizeEmail($email);
    }

    // Очистка URL
    public function sanitizeUrl(string $url): string
    {
        self::ensureSanitizationHelper();

        // Используем SanitizationHelper как единую точку входа
        return SanitizationHelper::sanitizeUrl($url);
    }
}
