<?php

declare(strict_types=1);

namespace Flowaxy\Infrastructure\Persistence\Repositories;

use Flowaxy\Domain\Theme\Services\ThemeSettingsRepositoryInterface;
use Flowaxy\Support\Facades\Log;
use Flowaxy\Support\Helpers\CacheHelper;
use Throwable;

final class ThemeSettingsRepository implements ThemeSettingsRepositoryInterface
{
    /**
     * Отримання налаштування теми за ключем
     *
     * @param string $themeSlug Slug теми
     * @param string $key Ключ налаштування
     * @param mixed $default Значення за замовчуванням
     * @return mixed
     */
    public function get(string $themeSlug, string $key, mixed $default = null): mixed
    {
        if ($themeSlug === '' || $key === '') {
            return $default;
        }

        try {
            if (class_exists(\Flowaxy\Infrastructure\Config\ThemeSettingsManager::class)) {
                return \Flowaxy\Infrastructure\Config\ThemeSettingsManager::getValue($themeSlug, $key, $default);
            }
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка отримання налаштування теми', ['exception' => $e]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
        }

        return $default;
    }

    /**
     * Отримання всіх налаштувань теми
     *
     * @param string $themeSlug Slug теми
     * @return array
     */
    public function getAll(string $themeSlug): array
    {
        if ($themeSlug === '') {
            return [];
        }

        try {
            if (class_exists(\Flowaxy\Infrastructure\Config\ThemeSettingsManager::class)) {
                return \Flowaxy\Infrastructure\Config\ThemeSettingsManager::getSettings($themeSlug);
            }
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка отримання налаштувань теми', ['exception' => $e]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
        }

        return [];
    }

    /**
     * Встановити налаштування теми
     *
     * @param string $themeSlug Slug теми
     * @param string $key Ключ налаштування
     * @param mixed $value Значення
     * @return bool
     */
    public function set(string $themeSlug, string $key, mixed $value): bool
    {
        if ($themeSlug === '' || $key === '') {
            return false;
        }

        try {
            if (class_exists(\Flowaxy\Infrastructure\Config\ThemeSettingsManager::class)) {
                $result = \Flowaxy\Infrastructure\Config\ThemeSettingsManager::setValue($themeSlug, $key, $value);
                if ($result) {
                    $this->clearCache($themeSlug);
                }
                return $result;
            }
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка збереження налаштування теми', ['exception' => $e]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
        }

        return false;
    }

    /**
     * Видалити налаштування теми
     *
     * @param string $themeSlug Slug теми
     * @param string $key Ключ налаштування
     * @return bool
     */
    public function delete(string $themeSlug, string $key): bool
    {
        if ($themeSlug === '' || $key === '') {
            return false;
        }

        try {
            if (class_exists(\Flowaxy\Infrastructure\Config\ThemeSettingsManager::class)) {
                $settings = \Flowaxy\Infrastructure\Config\ThemeSettingsManager::getSettings($themeSlug);
                if (isset($settings[$key])) {
                    unset($settings[$key]);
                    $result = \Flowaxy\Infrastructure\Config\ThemeSettingsManager::saveSettings($themeSlug, $settings);
                    if ($result) {
                        $this->clearCache($themeSlug);
                    }
                    return $result;
                }
            }
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка видалення налаштування теми', ['exception' => $e]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
        }

        return false;
    }

    /**
     * Очистити всі налаштування теми
     *
     * @param string $themeSlug Slug теми
     * @return bool
     */
    public function clear(string $themeSlug): bool
    {
        if ($themeSlug === '') {
            return false;
        }

        try {
            if (class_exists(\Flowaxy\Infrastructure\Config\ThemeSettingsManager::class)) {
                $result = \Flowaxy\Infrastructure\Config\ThemeSettingsManager::saveSettings($themeSlug, []);
                if ($result) {
                    $this->clearCache($themeSlug);
                }
                return $result;
            }
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка очищення налаштувань теми', ['exception' => $e]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
        }

        return false;
    }

    /**
     * @deprecated Используйте getAll() для получения всех настроек
     */
    public function getValue(string $themeSlug, string $key, mixed $default = null): mixed
    {
        try {
            if (class_exists(\Flowaxy\Infrastructure\Config\ThemeSettingsManager::class)) {
                return \Flowaxy\Infrastructure\Config\ThemeSettingsManager::getValue($themeSlug, $key, $default);
            }
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка отримання налаштування теми', ['exception' => $e]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
        }

        return $default;
    }

    /**
     * @deprecated Используйте set() для установки значения
     */
    public function setValue(string $themeSlug, string $key, mixed $value): bool
    {
        return $this->set($themeSlug, $key, $value);
    }

    // Збереження множини налаштувань теми
    public function setMany(string $themeSlug, array $settings): bool
    {
        if ($themeSlug === '') {
            return false;
        }

        try {
            if (class_exists(\Flowaxy\Infrastructure\Config\ThemeSettingsManager::class)) {
                $result = \Flowaxy\Infrastructure\Config\ThemeSettingsManager::saveSettings($themeSlug, $settings);
                if ($result) {
                    $this->clearCache($themeSlug);
                }
                return $result;
            }
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка збереження налаштувань теми', ['exception' => $e]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
        }

        return false;
    }

    public function clearCache(string $themeSlug): void
    {
        if ($themeSlug === '') {
            return;
        }

        CacheHelper::forget('theme_settings_' . $themeSlug);
    }
}
