<?php

/**
 * Клас для представлення email повідомлення
 *
 * @package Flowaxy\Infrastructure\Mail
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Infrastructure\Mail;

final class Message
{
    private string $to = '';
    private string $subject = '';
    private string $body = '';
    private ?string $from = null;
    private ?string $fromName = null;
    private array $headers = [];
    private array $attachments = [];
    private bool $isHtml = true;

    /**
     * Встановити отримувача
     *
     * @param string $to
     * @return self
     */
    public function to(string $to): self
    {
        $this->to = $to;
        return $this;
    }

    /**
     * Отримати отримувача
     *
     * @return string
     */
    public function getTo(): string
    {
        return $this->to;
    }

    /**
     * Встановити тему
     *
     * @param string $subject
     * @return self
     */
    public function subject(string $subject): self
    {
        $this->subject = $subject;
        return $this;
    }

    /**
     * Отримати тему
     *
     * @return string
     */
    public function getSubject(): string
    {
        return $this->subject;
    }

    /**
     * Встановити тіло повідомлення
     *
     * @param string $body
     * @return self
     */
    public function body(string $body): self
    {
        $this->body = $body;
        return $this;
    }

    /**
     * Отримати тіло повідомлення
     *
     * @return string
     */
    public function getBody(): string
    {
        return $this->body;
    }

    /**
     * Встановити відправника
     *
     * @param string $email
     * @param string|null $name
     * @return self
     */
    public function from(string $email, ?string $name = null): self
    {
        $this->from = $email;
        $this->fromName = $name;
        return $this;
    }

    /**
     * Отримати email відправника
     *
     * @return string|null
     */
    public function getFrom(): ?string
    {
        return $this->from;
    }

    /**
     * Отримати ім'я відправника
     *
     * @return string|null
     */
    public function getFromName(): ?string
    {
        return $this->fromName;
    }

    /**
     * Встановити заголовок
     *
     * @param string $name
     * @param string $value
     * @return self
     */
    public function header(string $name, string $value): self
    {
        $this->headers[$name] = $value;
        return $this;
    }

    /**
     * Отримати заголовки
     *
     * @return array<string, string>
     */
    public function getHeaders(): array
    {
        return $this->headers;
    }

    /**
     * Додати вкладення
     *
     * @param string $path
     * @param string|null $name
     * @return self
     */
    public function attach(string $path, ?string $name = null): self
    {
        $this->attachments[] = [
            'path' => $path,
            'name' => $name ?? basename($path),
        ];
        return $this;
    }

    /**
     * Отримати вкладення
     *
     * @return array<int, array<string, string>>
     */
    public function getAttachments(): array
    {
        return $this->attachments;
    }

    /**
     * Встановити HTML формат
     *
     * @param bool $isHtml
     * @return self
     */
    public function html(bool $isHtml = true): self
    {
        $this->isHtml = $isHtml;
        return $this;
    }

    /**
     * Перевірити чи це HTML
     *
     * @return bool
     */
    public function isHtml(): bool
    {
        return $this->isHtml;
    }
}
