<?php

declare(strict_types=1);

namespace Flowaxy\Infrastructure\Assets;

/**
 * Менеджер CDN для статических ресурсов
 *
 * @package Flowaxy\Infrastructure\Assets
 */
final class CdnManager
{
    private ?string $cdnUrl = null;
    private bool $enabled = false;

    public function __construct()
    {
        $this->cdnUrl = $_ENV['CDN_URL'] ?? null;
        $this->enabled = !empty($this->cdnUrl) && filter_var($this->cdnUrl, FILTER_VALIDATE_URL) !== false;
    }

    /**
     * Получить CDN URL для ресурса
     *
     * @param string $path Путь к ресурсу
     * @return string
     */
    public function url(string $path): string
    {
        if (!$this->enabled || $this->cdnUrl === null) {
            return $path;
        }

        // Убираем начальный слеш, если есть
        $path = ltrim($path, '/');

        // Объединяем CDN URL с путем
        return rtrim($this->cdnUrl, '/') . '/' . $path;
    }

    /**
     * Проверить, включен ли CDN
     *
     * @return bool
     */
    public function isEnabled(): bool
    {
        return $this->enabled;
    }

    /**
     * Установить CDN URL
     *
     * @param string $url
     * @return void
     */
    public function setUrl(string $url): void
    {
        $this->cdnUrl = $url;
        $this->enabled = !empty($url) && filter_var($url, FILTER_VALIDATE_URL) !== false;
    }
}
