<?php

declare(strict_types=1);

namespace Flowaxy\Domain\User\Services;

use Exception;
use Flowaxy\Domain\Events\DomainEventBridge;
use Flowaxy\Domain\User\Events\UserCreatedEvent;
use Flowaxy\Support\Facades\Hash;
use Flowaxy\Support\Facades\Log;
use Throwable;

// Сервіс реєстрації користувача
final class RegisterUserService
{
    public function __construct(
        private readonly UserService $userService
    ) {
    }

    public function execute(string $username, string $password): bool
    {
        if ($username === '' || $password === '') {
            try {
                Log::Warning('Порожній логін або пароль');
            } catch (Throwable $e) {
                // Ignore logging errors
            }
            return false;
        }

        $existingUser = $this->userService->findByUsername($username);
        if ($existingUser !== null) {
            try {
                Log::Warning('Користувач вже існує', ['username' => $username]);
            } catch (Throwable $e) {
                // Ignore logging errors
            }
            return false;
        }

        $passwordHash = Hash::make($password);
        $user = $this->userService->create($username, $passwordHash);

        if ($user !== null) {
            try {
                DomainEventBridge::dispatch(new UserCreatedEvent(
                    userId: $user->id,
                    username: $username,
                    email: ''
                ));
            } catch (Throwable $e) {
                // Ignore dispatch errors
            }
            return true;
        }

        $exception = new Exception('Failed to register user');
        try {
            Log::Error('Помилка реєстрації користувача', [
                'exception' => $exception,
            ]);
        } catch (Throwable $e) {
            // Ignore logging errors
        }
        return false;
    }
}
