<?php

declare(strict_types=1);

namespace Flowaxy\Domain\Theme\Services;

use Flowaxy\Domain\Theme\Services\ThemeRepositoryInterface;
use Flowaxy\Domain\Theme\Services\ThemeSettingsRepositoryInterface;
use Flowaxy\Support\Facades\Log;
use Throwable;

use function array_filter;
use const ARRAY_FILTER_USE_KEY;

// Сервіс оновлення налаштувань теми
final class UpdateThemeSettingsService
{
    public function __construct(
        private readonly ThemeSettingsRepositoryInterface $settings,
        private readonly ThemeRepositoryInterface $themes
    ) {
    }

    public function execute(string $themeSlug, array $settings): bool
    {
        if ($themeSlug === '' || empty($settings)) {
            try {
                Log::Warning('Невірний slug теми або порожні налаштування', ['theme_slug' => $themeSlug]);
            } catch (Throwable $e) {
                // Ignore logging errors
            }
            return false;
        }

        $theme = $this->themes->find($themeSlug);
        if ($theme === null) {
            try {
                Log::Error('Тему не знайдено', ['theme_slug' => $themeSlug]);
            } catch (Throwable $e) {
                // Ignore logging errors
            }
            return false;
        }

        $valid = array_filter(
            $settings,
            static fn ($key) => $key !== '',
            ARRAY_FILTER_USE_KEY
        );

        if (empty($valid)) {
            try {
                Log::Warning('Немає валідних налаштувань після фільтрації', ['theme_slug' => $themeSlug]);
            } catch (Throwable $e) {
                // Ignore logging errors
            }
            return false;
        }

        $result = $this->settings->setMany($themeSlug, $valid);
        if ($result) {
            $this->settings->clearCache($themeSlug);
        } else {
            try {
                Log::Error('Помилка оновлення налаштувань теми', ['theme_slug' => $themeSlug]);
            } catch (Throwable $e) {
                // Ignore logging errors
            }
        }

        return $result;
    }
}
