<?php

declare(strict_types=1);

namespace Flowaxy\Core\System\Registry;

/**
 * Наблюдатель за изменениями файлов
 *
 * @package Flowaxy\Core\System\Registry
 */
final class FileWatcher
{
    private array $watchedPaths = [];
    private array $lastModified = [];

    /**
     * Добавить путь для наблюдения
     *
     * @param string $path Путь к файлу или директории
     * @return void
     */
    public function watch(string $path): void
    {
        if (file_exists($path)) {
            $this->watchedPaths[] = $path;
            $this->lastModified[$path] = filemtime($path);
        }
    }

    /**
     * Проверить, изменились ли файлы
     *
     * @return array<int, string> Массив измененных путей
     */
    public function checkChanges(): array
    {
        $changed = [];

        foreach ($this->watchedPaths as $path) {
            if (!file_exists($path)) {
                continue;
            }

            $currentModified = filemtime($path);
            if (!isset($this->lastModified[$path]) || $currentModified > $this->lastModified[$path]) {
                $changed[] = $path;
                $this->lastModified[$path] = $currentModified;
            }
        }

        return $changed;
    }

    /**
     * Получить список наблюдаемых путей
     *
     * @return array<int, string>
     */
    public function getWatchedPaths(): array
    {
        return $this->watchedPaths;
    }
}
