<?php

/**
 * Сервіс для відображення сторінки встановлення
 *
 * @package Flowaxy\Modules\Errors\Services
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Core\System\Errors\Services;

use Flowaxy\Core\System\PathResolver;

use function class_exists;
use function dirname;
use function fastcgi_finish_request;
use function file_exists;
use function function_exists;
use function header;
use function headers_sent;
use function http_response_code;
use function is_readable;
use function ob_end_clean;
use function ob_get_level;
use function parse_url;
use function str_starts_with;
use Throwable;
use const DS;
use const PHP_URL_PATH;

final class InstallationService
{
    /**
     * Показує сторінку з повідомленням про необхідність установки системи
     *
     * @return never
     */
    public function show(): void
    {
        $requestUri = $_SERVER['REQUEST_URI'] ?? '/';
        $path = parse_url($requestUri, PHP_URL_PATH) ?? '/';

        // Дозволяємо доступ до встановлювача
        if (str_starts_with($path, '/install')) {
            return;
        }

        // Зупиняємо всі буфери виводу
        while (ob_get_level() > 0) {
            ob_end_clean();
        }

        // Встановлюємо код відповіді
        if (!headers_sent()) {
            http_response_code(503);
            header('Content-Type: text/html; charset=UTF-8');
            header('Cache-Control: no-cache, no-store, must-revalidate');
            header('Pragma: no-cache');
            header('Expires: 0');
        }

        // Шлях до шаблону
        $templateFile = $this->getTemplatePath();

        // Діагностика: перевіряємо, чи шаблон знайдено
        if (!file_exists($templateFile)) {
            // Якщо шаблон не знайдено через PathResolver, спробуємо відносний шлях
            $currentDir = __DIR__; // flowaxy/Modules/Errors/Services/
            $templatesDir = dirname($currentDir) . DS . 'Templates';
            $relativeTemplatePath = $templatesDir . DS . 'installation-required.php';

            if (file_exists($relativeTemplatePath)) {
                $templateFile = $relativeTemplatePath;
            }
        }

        if (file_exists($templateFile) && is_readable($templateFile)) {
            include $templateFile;
        } else {
            // Fallback якщо шаблон недоступний
            $this->renderFallback();
        }

        // Обов'язковий вихід
        if (function_exists('fastcgi_finish_request')) {
            fastcgi_finish_request();
        }
        exit(0);
    }

    /**
     * Отримання шляху до шаблону встановлення
     *
     * @return string
     */
    private function getTemplatePath(): string
    {
        // Спочатку шукаємо в модулі Errors (відносний шлях від поточного файлу)
        $currentDir = __DIR__; // flowaxy/Modules/Errors/Services/
        $templatesDir = dirname($currentDir) . DS . 'Templates';
        $moduleTemplatePath = $templatesDir . DS . 'installation-required.php';

        if (file_exists($moduleTemplatePath)) {
            return $moduleTemplatePath;
        }

        // Fallback через PathResolver
        try {
            $moduleTemplatePath = PathResolver::flowaxy() . DS .
                'Core' . DS .
                'System' . DS .
                'Errors' . DS .
                'Templates' . DS .
                'installation-required.php';

            if (file_exists($moduleTemplatePath)) {
                return $moduleTemplatePath;
            }
        } catch (Throwable $e) {
            // PathResolver не доступний - продовжуємо
        }

        // Fallback на старий шлях
        try {
            return PathResolver::engine() . DS .
                'interface' . DS .
                'errors' . DS .
                'exceptions' . DS .
                'installation-required.php';
        } catch (Throwable $e) {
            // Якщо PathResolver не доступний - повертаємо відносний шлях
            return $moduleTemplatePath;
        }
    }

    /**
     * Fallback рендеринг
     *
     * @return void
     */
    private function renderFallback(): void
    {
        // Мінімальне повідомлення, якщо шаблон недоступний
        if (!\headers_sent()) {
            \http_response_code(503);
            \header('Content-Type: text/html; charset=UTF-8');
        }
        echo '<!DOCTYPE html><html><head><meta charset="UTF-8"><title>Потрібна установка</title></head><body>';
        echo '<h1>Потрібна установка системи</h1>';
        echo '<p>База даних не налаштована. Для початку роботи необхідно встановити Flowaxy CMS.</p>';
        echo '<p><a href="/install">Встановити систему</a></p>';
        echo '</body></html>';
    }
}
