<?php

/**
 * Сервіс для рендерингу шаблонів помилок
 *
 * @package Flowaxy\Modules\Errors\Services
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Core\System\Errors\Services;

use Flowaxy\Core\System\PathResolver;
use Throwable;

use function class_exists;
use function dirname;
use function extract;
use function file_exists;
use const DS;
use const EXTR_SKIP;

final class ErrorRenderer
{
    /**
     * Рендеринг шаблону помилки
     *
     * @param string $templateName Ім'я шаблону (без розширення)
     * @param array<string, mixed> $data Дані для передачі в шаблон
     * @return void
     */
    public function render(string $templateName, array $data = []): void
    {
        $templateFile = $this->getTemplatePath($templateName);

        if (file_exists($templateFile)) {
            // Екстрагуємо змінні для використання в шаблоні
            extract($data, EXTR_SKIP);
            include $templateFile;
        } else {
            // Fallback якщо шаблон не знайдено
            $this->renderFallback($data);
        }
    }

    /**
     * Отримання шляху до шаблону
     *
     * @param string $templateName Ім'я шаблону
     * @return string
     */
    private function getTemplatePath(string $templateName): string
    {
        // Спочатку шукаємо в модулі Errors (відносний шлях від поточного файлу)
        $currentDir = __DIR__; // flowaxy/Modules/Errors/Services/
        $templatesDir = dirname($currentDir) . DS . 'Templates';
        $moduleTemplatePath = $templatesDir . DS . $templateName . '.php';

        if (file_exists($moduleTemplatePath)) {
            return $moduleTemplatePath;
        }

        // Fallback через PathResolver
        try {
            $moduleTemplatePath = PathResolver::flowaxy() . DS .
                'Core' . DS .
                'System' . DS .
                'Errors' . DS .
                'Templates' . DS .
                $templateName . '.php';

            if (file_exists($moduleTemplatePath)) {
                return $moduleTemplatePath;
            }
        } catch (Throwable $e) {
            // PathResolver не доступний - продовжуємо
        }

        // Fallback на старий шлях
        try {
            return PathResolver::engine() . DS .
                'interface' . DS .
                'errors' . DS .
                $templateName . '.php';
        } catch (\Throwable $e) {
            // Якщо PathResolver не доступний - повертаємо відносний шлях
            return $moduleTemplatePath;
        }
    }

    /**
     * Fallback рендеринг якщо шаблон не знайдено
     *
     * @param array<string, mixed> $data
     * @return void
     */
    private function renderFallback(array $data): void
    {
        $httpCode = $data['httpCode'] ?? 500;
        $title = $data['errorTitle'] ?? 'Помилка';
        $message = $data['errorMessage'] ?? 'Сталася невідома помилка';

        echo '<!DOCTYPE html><html><head><meta charset="UTF-8"><title>' . $httpCode . ' - ' . \htmlspecialchars($title, ENT_QUOTES, 'UTF-8') . '</title></head><body>';
        echo '<h1>' . $httpCode . ' - ' . \htmlspecialchars($title, ENT_QUOTES, 'UTF-8') . '</h1>';
        echo '<p>' . \htmlspecialchars($message, ENT_QUOTES, 'UTF-8') . '</p>';
        echo '</body></html>';
    }
}
