<?php

/**
 * Базовий клас для системних модулів
 *
 * @package Flowaxy\Core\System\Base
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Core\System\Base;

abstract class BaseModule implements ModuleInterface
{
    /**
     * @var array<string, static>
     */
    protected static array $instances = [];

    /**
     * Конструктор
     */
    protected function __construct()
    {
        $this->init();
    }

    /**
     * Ініціалізація модуля (перевизначається в дочірніх класах)
     */
    protected function init(): void
    {
        // Перевизначається в дочірніх класах
    }

    /**
     * Отримання екземпляра модуля (Singleton)
     *
     * @return static
     */
    public static function getInstance()
    {
        $className = static::class;
        if (!isset(self::$instances[$className])) {
            self::$instances[$className] = new static();
        }

        return self::$instances[$className];
    }

    /**
     * Реєстрація хуків модуля
     * Викликається автоматично при завантаженні модуля
     */
    public function registerHooks(): void
    {
        // Перевизначається в дочірніх класах
    }

    /**
     * Отримання інформації про модуль
     *
     * @return array<string, mixed>
     */
    abstract public function getInfo(): array;

    /**
     * Отримання API методів модуля
     *
     * @return array<string, mixed> Масив з описом доступних методів
     */
    public function getApiMethods(): array
    {
        return [];
    }

    /**
     * Реєстрація модуля (маршрути, хуки, сервіси)
     */
    public function register(): void
    {
        $this->registerHooks();
    }

    /**
     * Ініціалізація модуля після реєстрації
     */
    public function boot(): void
    {
        // Перевизначається в дочірніх класах
    }
}
