<?php

/**
 * Менеджер маршрутизації системи
 * Визначає контекст (адмінка / API / публічна частина) та управляє маршрутами
 *
 * @package Flowaxy\Core\Routing
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Core\Routing;

use Flowaxy\Core\Routing\Router;
use Flowaxy\Core\System\PathResolver;
use Flowaxy\Infrastructure\Cache\Cache as CacheInstance;
use Flowaxy\Infrastructure\Persistence\Database\Database;
use Flowaxy\Infrastructure\Security\RequestFilter;
use Flowaxy\Interface\Http\AjaxHandler;
use Flowaxy\Interface\UI\SimpleTemplate;
use Flowaxy\Core\System\Errors\ErrorModuleLoader;
use Flowaxy\Support\Facades\Hooks;
use Flowaxy\Support\Facades\Log;
use Flowaxy\Support\Facades\Session;
use Flowaxy\Support\Managers\ThemeManager;
use Exception;
use Throwable;

use const DIRECTORY_SEPARATOR as DS;

use function class_exists;
use function file_exists;
use function function_exists;
use function is_readable;
use function ob_end_clean;
use function ob_get_level;
use function parse_url;
use function str_starts_with;

final class RouterManager
{
    public const CONTEXT_ADMIN = 'admin';
    public const CONTEXT_API = 'api';
    public const CONTEXT_PUBLIC = 'public';

    private static ?self $instance = null;

    private Router $router;
    private string $context = self::CONTEXT_PUBLIC;
    private ?string $basePath = null;
    private array $routeCache = [];
    private bool $routesLoaded = false;

    /**
     * @throws Exception
     */
    private function __construct()
    {
        // Завантажуємо Facade та Hooks перед використанням
        $this->loadFacades();

        $this->detectContext();
        $this->bootRouter();
        $this->registerCoreRoutes();
    }

    /**
     * Завантаження фасадів, необхідних для RouterManager
     *
     * @return void
     */
    private function loadFacades(): void
    {
        $flowaxyDir = PathResolver::flowaxy();

        // Завантажуємо базовий клас Facade
        $facadeFile = $flowaxyDir . DS . 'Support' . DS . 'Facades' . DS . 'Facade.php';
        if (!class_exists('Flowaxy\Support\Facades\Facade') && file_exists($facadeFile) && is_readable($facadeFile)) {
            require_once $facadeFile;
        }

        // Завантажуємо Hooks facade
        $hooksFile = $flowaxyDir . DS . 'Support' . DS . 'Facades' . DS . 'Hooks.php';
        if (!class_exists('Flowaxy\Support\Facades\Hooks') && file_exists($hooksFile) && is_readable($hooksFile)) {
            require_once $hooksFile;
        }
    }

    /**
     * Отримання екземпляра (Singleton)
     *
     * @return self
     * @throws Exception
     */
    public static function getInstance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Отримання об'єкта Router
     *
     * @return Router
     */
    public function getRouter(): Router
    {
        return $this->router;
    }

    /**
     * Отримання поточного контексту
     *
     * @return string
     */
    public function getContext(): string
    {
        return $this->context;
    }

    /**
     * Перевірка, чи є поточний контекст адмінкою
     *
     * @return bool
     */
    public function isAdmin(): bool
    {
        return $this->context === self::CONTEXT_ADMIN;
    }

    /**
     * Перевірка, чи є поточний контекст API
     *
     * @return bool
     */
    public function isApi(): bool
    {
        return $this->context === self::CONTEXT_API;
    }

    /**
     * Перевірка, чи є поточний контекст публічною частиною
     *
     * @return bool
     */
    public function isPublic(): bool
    {
        return $this->context === self::CONTEXT_PUBLIC;
    }

    /**
     * Отримання базового шляху для поточного контексту
     *
     * @return string
     */
    public function getBasePath(): string
    {
        return $this->basePath ?? '/';
    }

    /**
     * Виконання диспетчеризації маршрутів
     *
     * @return void
     */
    public function dispatch(): void
    {
        // Инсталлятор удален - проверка /install маршрута больше не нужна

        // Обробка AJAX запитів
        // Завантажуємо AjaxHandler та його залежності
        if (!class_exists('Flowaxy\Interface\Http\AjaxHandler')) {
            $flowaxyDir = PathResolver::flowaxy();

            // Спочатку завантажуємо інтерфейс (обов'язкова залежність)
            if (!interface_exists('Flowaxy\Contracts\Http\AjaxHandlerInterface')) {
                // Спробуємо через автозавантажувач
                spl_autoload_call('Flowaxy\Contracts\Http\AjaxHandlerInterface');

                // Якщо не спрацював, завантажуємо вручну
                if (!interface_exists('Flowaxy\Contracts\Http\AjaxHandlerInterface')) {
                    $interfaceFile = $flowaxyDir . DS . 'Contracts' . DS . 'Http' . DS . 'AjaxHandlerInterface.php';
                    if (file_exists($interfaceFile) && is_readable($interfaceFile)) {
                        require_once $interfaceFile;
                    }
                }
            }

            // Потім завантажуємо клас
            // Спробуємо через автозавантажувач
            spl_autoload_call('Flowaxy\Interface\Http\AjaxHandler');

            // Якщо не спрацював, завантажуємо вручну
            if (!class_exists('Flowaxy\Interface\Http\AjaxHandler')) {
                $ajaxHandlerFile = $flowaxyDir . DS . 'Interface' . DS . 'Http' . DS . 'AjaxHandler.php';
                if (file_exists($ajaxHandlerFile) && is_readable($ajaxHandlerFile)) {
                    require_once $ajaxHandlerFile;
                }
            }
        }

        if (class_exists('Flowaxy\Interface\Http\AjaxHandler') && AjaxHandler::isAjax()) {
            while (ob_get_level() > 0) {
                ob_end_clean();
            }
            ini_set('display_errors', '0');
        }

        // Завантажуємо додаткові маршрути з плагінів, якщо ще не завантажені
        if (! $this->routesLoaded) {
            $this->loadPluginRoutes();
            $this->routesLoaded = true;
        }

        $this->router->dispatch();
    }

    /**
     * Додавання маршруту в поточний контекст
     *
     * @param string|array $methods HTTP методи (GET, POST, PUT, DELETE тощо)
     * @param string $path Шлях маршруту
     * @param callable|string $handler Обробник (функція, замикання, клас)
     * @param array $options Додаткові опції (middleware, name, тощо)
     * @return self
     */
    public function addRoute($methods, string $path, $handler, array $options = []): self
    {
        $this->router->add($methods, $path, $handler, $options);

        return $this;
    }

    /**
     * Реєстрація сторінки адмінки у вигляді класу
     *
     * @param string $slug Slug сторінки
     * @param string $className Ім'я класу сторінки
     * @param array $options Додаткові опції
     * @return void
     */
    public function registerAdminPage(string $slug, string $className, array $options = []): void
    {
        if ($this->context !== self::CONTEXT_ADMIN) {
            return;
        }
        $methods = $options['methods'] ?? ['GET', 'POST'];
        $this->router->add($methods, $slug, $className, $options);
    }

    /**
     * Виконання callback тільки в певному контексті
     *
     * @param string $context Контекст (CONTEXT_ADMIN, CONTEXT_API, CONTEXT_PUBLIC)
     * @param callable $callback Callback функція
     * @return void
     */
    public function when(string $context, callable $callback): void
    {
        if ($this->context === $context) {
            $callback($this->router, $this);
        }
    }

    /**
     * Визначення контексту на основі URI
     * Оптимізована версія з покращеною обробкою шляхів
     *
     * @return void
     */
    private function detectContext(): void
    {
        $uri = RequestFilter::server('REQUEST_URI', '/');
        $path = parse_url($uri, PHP_URL_PATH) ?? '/';

        // Нормалізація шляху
        $path = $this->normalizePath($path);

        // Перевіряємо контексти в порядку пріоритету
        // 1. API (найвищий пріоритет)
        if ($this->isApiPath($path)) {
            $this->context = self::CONTEXT_API;
            $this->basePath = $this->detectApiBasePath($path);

            return;
        }

        // 2. Адмінка
        if ($this->isAdminPath($path)) {
            $this->context = self::CONTEXT_ADMIN;
            $this->basePath = '/admin';

            return;
        }

        // 3. Публічна частина (за замовчуванням)
        $this->context = self::CONTEXT_PUBLIC;
        $this->basePath = '/';
    }

    /**
     * Нормалізація шляху (видалення дублюючих слэшів, очищення)
     *
     * @param string $path Шлях для нормалізації
     * @return string Нормалізований шлях
     */
    private function normalizePath(string $path): string
    {
        // Видалення дублюючих слэшів (оптимізовано)
        $path = preg_replace('#/+#', '/', $path);

        // Видалення кінцевого слэша (крім кореня)
        if ($path !== '/' && str_ends_with($path, '/')) {
            $path = rtrim($path, '/');
        }

        // Переконуємося, що шлях починається зі слэша
        if (! str_starts_with($path, '/')) {
            $path = '/' . $path;
        }

        return $path;
    }

    /**
     * Перевірка, чи є шлях API шляхом
     *
     * @param string $path Шлях для перевірки
     * @return bool
     */
    private function isApiPath(string $path): bool
    {
        // Перевіряємо різні варіанти API шляхів
        // /api, /api/, /api/v1, /api/v2, /api-v1 тощо
        return preg_match('#^/api(/v\d+)?(/|$)#', $path) === 1 ||
               preg_match('#^/api-v?\d+#', $path) === 1;
    }

    /**
     * Визначення базового шляху для API
     *
     * @param string $path Шлях запиту
     * @return string Базовий шлях API
     */
    private function detectApiBasePath(string $path): string
    {
        // Перевіряємо версію API в шляху /api/v1, /api/v2 тощо
        if (preg_match('#^/api/v(\d+)#', $path, $matches)) {
            return '/api/v' . $matches[1];
        }

        // Перевіряємо старий формат api-v1, api-v2 тощо
        if (preg_match('#^/api-v(\d+)#', $path, $matches)) {
            return '/api/v' . $matches[1];
        }

        // За замовчуванням використовуємо v1
        return '/api/v1';
    }

    /**
     * Перевірка, чи є шлях адмінським шляхом
     *
     * @param string $path Шлях для перевірки
     * @return bool
     */
    private function isAdminPath(string $path): bool
    {
        // Точне співпадіння /admin або шлях починається з /admin/
        return $path === '/admin' || str_starts_with($path, '/admin/');
    }

    /**
     * Ініціалізація Router з потрібними параметрами
     *
     * @return void
     * @throws \RuntimeException
     */
    private function bootRouter(): void
    {
        if (! class_exists(Router::class)) {
            throw new \RuntimeException('Клас Router не знайдено');
        }

        // Визначаємо маршрут за замовчуванням для адмінки
        $defaultRoute = $this->context === self::CONTEXT_ADMIN ? 'dashboard' : null;

        $this->router = new Router($this->basePath, $defaultRoute);
    }

    /**
     * Реєстрація основних маршрутів залежно від контексту
     *
     * @return void
     */
    private function registerCoreRoutes(): void
    {
        match ($this->context) {
            self::CONTEXT_API => $this->loadApiRoutes(),
            self::CONTEXT_ADMIN => $this->loadAdminRoutes(),
            default => $this->registerPublicRoutes(),
        };
    }

    /**
     * Завантаження маршрутів API
     *
     * @return void
     */
    private function loadApiRoutes(): void
    {
        $routesFile = PathResolver::bootstrap() . DS . 'api-routes.php';
        if (file_exists($routesFile)) {
            $router = $this->router;
            require $routesFile;
        }
    }

    /**
     * Завантаження маршрутів адмінки
     *
     * @return void
     */
    private function loadAdminRoutes(): void
    {
        // SCSS компіляція НЕ виконується під час HTTP-запитів.
        // Компіляція стилів переноситься в CLI-команди.

        // Перевірка стану системи перед завантаженням маршрутів адмінки
        // Перевіряємо доступність БД та наявність обов'язкових таблиць
        if (function_exists('checkSystemState')) {
            checkSystemState();
        }

        // Завантажуємо SimpleTemplate якщо він існує
        $simpleTemplate = PathResolver::flowaxy() . DS . 'Interface' . DS . 'UI' . DS . 'SimpleTemplate.php';
        if (file_exists($simpleTemplate)) {
            require_once $simpleTemplate;
        }

        // Регистрируем middleware для авторизации админ-страниц
        $this->registerAdminAuthMiddleware();

        // Завантажуємо маршрути адмінки
        // Новая структура: content/admin/includes/admin-routes.php
        $routesFile = PathResolver::contentAdmin() . DS . 'includes' . DS . 'admin-routes.php';

        // Fallback: старая структура (admin/Interface/AdminUI/includes/)
        if (!file_exists($routesFile)) {
            $routesFile = PathResolver::admin() . DS . 'Interface' . DS . 'AdminUI' . DS . 'includes' . DS . 'admin-routes.php';
        }

        // Fallback: старая структура (engine/interface/admin-ui/includes/)
        if (!file_exists($routesFile)) {
            $enginePath = PathResolver::engine();
            $routesFile = $enginePath . DS . 'interface' . DS . 'admin-ui' . DS . 'includes' . DS . 'admin-routes.php';
        }

        // Файл маршрутів повинен бути в правильному місці

        if (\file_exists($routesFile) && \is_readable($routesFile)) {
            $router = $this->router;
            require $routesFile;
        }

        // Дозволяємо плагінам додавати свої маршрути через хук
        try {
            Hooks::dispatch('admin_routes', $this->router, $this);
        } catch (Exception $e) {
            // Ignore hook errors
        }
    }

    /**
     * Реєстрація маршрутів плагіна з масиву
     * Завантажує сторінки та реєструє роути
     *
     * @param array<int, array<string, mixed>> $routes Масив роутів
     * @param string $pagesDir Директорія зі сторінками плагіна
     * @return void
     */
    public function registerPluginRoutes(array $routes, string $pagesDir): void
    {
        if ($this->router === null) {
            return;
        }

        // Завантажуємо сторінки перед реєстрацією роутів
        if (is_dir($pagesDir)) {
            $pageFiles = glob($pagesDir . DS . '*.php');
            foreach ($pageFiles as $pageFile) {
                if (file_exists($pageFile) && is_readable($pageFile)) {
                    require_once $pageFile;
                }
            }
        }

        // Реєструємо кожен роут
        foreach ($routes as $route) {
            if (is_array($route) && isset($route['methods'], $route['path'], $route['handler'])) {
                try {
                    $methods = $route['methods'];
                    $path = $route['path'];
                    $handler = $route['handler'];
                    $options = [
                        'middleware' => $route['middleware'] ?? [],
                        'name' => $route['name'] ?? null,
                    ];

                    $this->router->add($methods, $path, $handler, $options);
                } catch (\Throwable $e) {
                    // Логуємо помилку реєстрації маршруту
                    if (class_exists('\Flowaxy\Support\Facades\Log')) {
                        try {
                            \Flowaxy\Support\Facades\Log::Error('RouterManager: Помилка реєстрації маршруту плагіна', [
                                'exception' => $e->getMessage(),
                                'route' => $route,
                                'pages_dir' => $pagesDir,
                            ]);
                        } catch (\Throwable $logError) {
                            // Ignore logging errors
                        }
                    }
                }
            }
        }
    }

    /**
     * Регистрация middleware для авторизации админ-страниц
     * Применяется автоматически ко всем маршрутам кроме login/logout
     *
     * @return void
     */
    private function registerAdminAuthMiddleware(): void
    {
        $middlewareClass = 'Flowaxy\Core\Middleware\AdminAuthMiddleware';

        // Загружаем middleware класс
        if (!class_exists($middlewareClass)) {
            $middlewareFile = PathResolver::flowaxy() . DS . 'Core' . DS . 'Middleware' . DS . 'AdminAuthMiddleware.php';
            if (file_exists($middlewareFile) && is_readable($middlewareFile)) {
                require_once $middlewareFile;
            }
        }

        // Регистрируем middleware в роутере
        if (class_exists($middlewareClass) && $this->router !== null) {
            $this->router->middleware('admin.auth', [$middlewareClass, 'check']);
        }
    }

    /**
     * Завантаження маршрутів з плагінів
     *
     * @return void
     */
    private function loadPluginRoutes(): void
    {
        // Хук для додавання маршрутів плагінами
        try {
            Hooks::dispatch('register_routes', $this->router, $this);
        } catch (Exception $e) {
            // Ignore hook errors
        }
    }

    /**
     * Реєстрація публічних маршрутів
     *
     * @return void
     */
    private function registerPublicRoutes(): void
    {
        // Перевірка стану системи перед завантаженням публічних маршрутів
        // Якщо БД недоступна - показуємо 500 помилку замість публічної частини
        if (function_exists('checkSystemState')) {
            checkSystemState();
        }

        // Ініціалізуємо тему ПЕРЕД реєстрацією публічних маршрутів,
        // щоб вона була доступна для всіх публічних роутів (включаючи роути плагінів)
        $this->initializePublicTheme();

        // Ініціалізуємо плагіни ПЕРЕД реєстрацією публічних маршрутів,
        // щоб вони встигли підписатися на хук public_routes
        if (function_exists('pluginManager')) {
            try {
                $pluginManager = pluginManager();
                if ($pluginManager) {
                    // Спочатку завантажуємо плагіни (якщо ще не завантажені)
                    if (method_exists($pluginManager, 'loadPlugins')) {
                        $pluginManager->loadPlugins('public_routes');
                    }
                    // Потім ініціалізуємо їх
                    if (method_exists($pluginManager, 'initializePlugins')) {
                        $pluginManager->initializePlugins();
                    }
                }
            } catch (Exception $e) {
                // Ігноруємо помилки ініціалізації плагінів
            }
        }

        // Дозволяємо плагінам додавати свої публічні маршрути через хук
        // ВАЖЛИВО: викликаємо ПЕРЕД основним маршрутом '', щоб специфічні маршрути плагінів
        // реєструвалися раніше і мали пріоритет
        try {
            Hooks::dispatch('public_routes', $this->router, $this);
        } catch (Exception $e) {
            if (class_exists('\Flowaxy\Support\Facades\Log')) {
                try {
                    \Flowaxy\Support\Facades\Log::Error('RouterManager: Error dispatching public_routes hook', [
                        'exception' => $e->getMessage(),
                    ]);
                } catch (\Throwable $logError) {
                    // Ignore
                }
            }
        }

        // Основний маршрут для публічної частини (реєструється останнім як fallback)
        $this->router->add(['GET', 'POST'], '', function () {
            // Завантажуємо ThemeManager якщо функція ще не доступна
            if (!function_exists('themeManager')) {
                $themeManagerFile = PathResolver::engine() . DS . 'Support' . DS . 'Managers' . DS . 'ThemeManager.php';
                if (file_exists($themeManagerFile)) {
                    require_once $themeManagerFile;
                }
            }

            // Перевірка стану системи перед обробкою публічного маршруту
            if (function_exists('checkSystemState')) {
                checkSystemState();
            }

            // Примусово очищаємо кеш
            try {
                $cache = CacheInstance::getInstance();
                if ($cache !== null) {
                    $cache->delete('active_theme_slug');
                    $cache->delete('active_theme');
                }
            } catch (Exception $e) {
                // Кеш може бути ще не ініціалізований
            }

            if (class_exists(ThemeManager::class)) {
                $tm = ThemeManager::getManagerInstance();

                // Примусово перезавантажуємо активну тему
                $tm->reloadActiveTheme();

                $active = $tm->getActiveTheme();

                // Якщо активна тема не завантажена, спробуємо перезавантажити
                if (empty($active) || ! isset($active['slug']) || empty($active['slug'])) {
                    // Використовуємо метод перезавантаження
                    $tm->reloadActiveTheme();
                    $active = $tm->getActiveTheme();
                } else {
                    // Fallback: очищаємо кеш та спробуємо отримати з БД
                    try {
                        $cache = CacheInstance::getInstance();
                        if ($cache !== null) {
                            $cache->delete('active_theme_slug');
                            $cache->delete('active_theme');
                        }
                    } catch (Exception $e) {
                        // Кеш може бути ще не ініціалізований
                    }
                    try {
                        // Читаем активную тему из storage/config/system/themes.ini (статус activated)
                        $activeSlug = null;
                        if (class_exists(\Flowaxy\Infrastructure\Config\ThemeConfigManager::class)) {
                            $activeSlug = \Flowaxy\Infrastructure\Config\ThemeConfigManager::getActive();
                        }

                        if ($activeSlug && ! empty($activeSlug)) {
                            $active = $tm->getTheme($activeSlug);
                            if ($active && isset($active['slug'])) {
                                // Оновлюємо активну тему в менеджері через Reflection
                                try {
                                    $reflection = new \ReflectionClass($tm);
                                    $property = $reflection->getProperty('activeTheme');
                                    $property->setAccessible(true);
                                    $property->setValue($tm, $active);
                                } catch (Exception $e) {
                                    try {
                                        Log::Error('RouterManager: Error updating active theme', [
                                            'error' => $e->getMessage(),
                                            'exception' => $e,
                                        ]);
                                    } catch (Throwable $logError) {
                                        // Ignore logging errors
                                    }
                                }
                            }
                        }
                    } catch (Throwable $e) {
                        // Database connection error - ignore
                    }
                }

                // Повторна перевірка після спроби перезавантаження
                $active = $tm->getActiveTheme();

                if ($active && isset($active['slug']) && ! empty($active['slug'])) {
                    // Визначаємо константу для перевірки в темі
                    if (! \defined('FLOWAXY_CMS')) {
                        \define('FLOWAXY_CMS', true);
                    }

                    // Завантажуємо тему через ThemeLoader
                    if (class_exists(\Flowaxy\Support\Managers\ThemeLoader::class)) {
                        try {
                            $themesDir = \Flowaxy\Core\System\PathResolver::themes();
                            \Flowaxy\Support\Managers\ThemeLoader::initialize($themesDir);

                            // Загружаем тему через ThemeLoader (это создаст контейнер)
                            $theme = \Flowaxy\Support\Managers\ThemeLoader::load($active['slug']);
                            if ($theme !== null) {
                                // Устанавливаем активную тему в Theme API после загрузки (контейнер уже создан)
                                if (class_exists(\Flowaxy\Support\Theme\Theme::class)) {
                                    try {
                                        \Flowaxy\Support\Theme\Theme::setActive($active['slug']);
                                    } catch (\Throwable $e) {
                                        // Игнорируем ошибки установки активной темы
                                    }
                                }

                                // Викликаємо метод renderTheme() якщо він є, інакше використовуємо стандартний рендеринг
                                if (method_exists($theme, 'renderTheme')) {
                                    $theme->renderTheme();
                                } else {
                                    // Fallback: стандартний рендеринг через Theme API
                                    if (class_exists(\Flowaxy\Support\Theme\Theme::class)) {
                                        \Flowaxy\Support\Theme\Theme::render('index');
                                    }
                                }
                                return true;
                            }
                        } catch (Throwable $e) {
                            try {
                                Log::Warning("RouterManager: Theme render error", [
                                    'slug' => $active['slug'],
                                    'exception' => $e->getMessage(),
                                ]);
                            } catch (Throwable $logError) {
                                // Ignore logging errors
                            }
                        }
                    } else {
                        try {
                            Log::Warning("RouterManager: ThemeLoader not found", [
                                'slug' => $active['slug'],
                            ]);
                        } catch (Throwable $logError) {
                            // Ignore logging errors
                        }
                    }
                }
            }

            // Если тема не активирована, показываем сообщение об активации через стандартный дизайн (модуль Errors)
            if (function_exists('showHttpError')) {
                // Используем стандартный дизайн из админки (модуль Errors)
                showHttpError(200, 'Тема не активована', 'Для відображення сайту необхідно активувати тему в розділі Дизайн. Перейдіть в адмін-панель та активуйте одну з доступних тем.');
                return true;
            }

            // Fallback если showHttpError недоступна
            if (!headers_sent()) {
                http_response_code(200);
                header('Content-Type: text/html; charset=UTF-8');
            }
            echo 'Тема не активована. Для відображення сайту необхідно активувати тему в розділі Дизайн.';
            return true;
        });
    }

    /**
     * Ініціалізація теми для публічної частини
     * Викликається перед виконанням публічних роутів
     *
     * @return void
     */
    private function initializePublicTheme(): void
    {
        // Перевіряємо, чи тема вже ініціалізована
        static $themeInitialized = false;
        if ($themeInitialized) {
            return;
        }

        // Завантажуємо ThemeManager якщо функція ще не доступна
        if (!function_exists('themeManager')) {
            $themeManagerFile = PathResolver::engine() . DS . 'Support' . DS . 'Managers' . DS . 'ThemeManager.php';
            if (file_exists($themeManagerFile)) {
                require_once $themeManagerFile;
            }
        }

        if (!class_exists(ThemeManager::class)) {
            return;
        }

        try {
            $tm = ThemeManager::getManagerInstance();
            $tm->reloadActiveTheme();
            $active = $tm->getActiveTheme();

            // Якщо активна тема не завантажена, спробуємо перезавантажити
            if (empty($active) || !isset($active['slug']) || empty($active['slug'])) {
                $tm->reloadActiveTheme();
                $active = $tm->getActiveTheme();
            }

            // Повторна перевірка після спроби перезавантаження
            $active = $tm->getActiveTheme();

            if ($active && isset($active['slug']) && !empty($active['slug'])) {
                // Визначаємо константу для перевірки в темі
                if (!\defined('FLOWAXY_CMS')) {
                    \define('FLOWAXY_CMS', true);
                }

                // Завантажуємо тему через ThemeLoader
                if (class_exists(\Flowaxy\Support\Managers\ThemeLoader::class)) {
                    $themesDir = \Flowaxy\Core\System\PathResolver::themes();
                    \Flowaxy\Support\Managers\ThemeLoader::initialize($themesDir);

                    // Загружаем тему через ThemeLoader (это создаст контейнер)
                    $theme = \Flowaxy\Support\Managers\ThemeLoader::load($active['slug']);
                    if ($theme !== null) {
                        // Устанавливаем активную тему в Theme API после загрузки (контейнер уже создан)
                        if (class_exists(\Flowaxy\Support\Theme\Theme::class)) {
                            try {
                                \Flowaxy\Support\Theme\Theme::setActive($active['slug']);
                                $themeInitialized = true;
                            } catch (\Throwable $e) {
                                // Игнорируем ошибки установки активной темы
                            }
                        }
                    }
                }
            }
        } catch (\Throwable $e) {
            // Игнорируем ошибки инициализации темы
        }
    }

    /**
     * Очищення кешу маршрутів
     *
     * @return void
     */
    public function clearCache(): void
    {
        $this->routeCache = [];
        $this->routesLoaded = false;
    }

    /**
     * Отримання списку всіх зареєстрованих маршрутів (для відлагодження)
     *
     * @return array
     */
    public function getRoutes(): array
    {
        // Якщо Router має метод для отримання маршрутів
        if (method_exists($this->router, 'getRoutes')) {
            return $this->router->getRoutes();
        }

        return [];
    }

}
