<?php

/**
 * Интерфейс для системы автозагрузки классов
 *
 * Определяет контракт для всех реализаций автозагрузки в системе
 *
 * @package Flowaxy\Core\Contracts
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Core\Contracts;

interface AutoloaderInterface
{
    /**
     * Регистрация автозагрузчика в SPL
     *
     * @param bool $prepend Если true, добавляет в начало стека автозагрузчиков
     * @return void
     */
    public function register(bool $prepend = false): void;

    /**
     * Отмена регистрации автозагрузчика из SPL
     *
     * @return void
     */
    public function unregister(): void;

    /**
     * Добавление карты классов
     *
     * @param array<string, string> $classMap Массив [className => filePath]
     * @return void
     */
    public function addClassMap(array $classMap): void;

    /**
     * Добавление директории для поиска классов
     *
     * @param string $directory Путь к директории
     * @return void
     */
    public function addDirectory(string $directory): void;

    /**
     * Добавление нескольких директорий для поиска классов
     *
     * @param array<string> $directories Массив путей к директориям
     * @return void
     */
    public function addDirectories(array $directories): void;

    /**
     * Загрузка класса по имени
     *
     * @param string $className Полное имя класса (с namespace)
     * @return bool true если класс успешно загружен, false в противном случае
     */
    public function loadClass(string $className): bool;

    /**
     * Получение статистики работы автозагрузчика
     *
     * @return array<string, mixed> Статистика (map_hits, dir_hits, misses, loaded)
     */
    public function getStats(): array;

    /**
     * Установка логгера для автозагрузчика
     *
     * @param LoggerInterface|null $logger Экземпляр логгера
     * @return void
     */
    public function setLogger(?LoggerInterface $logger): void;

    /**
     * Включение/отключение логирования отсутствующих классов
     *
     * @param bool $enabled Включить или отключить
     * @return void
     */
    public function enableMissingClassLogging(bool $enabled = true): void;
}
