<?php

declare(strict_types=1);

namespace Flowaxy\Bootstrap;

use Flowaxy\Support\Helpers\DatabaseHelper;
use PDO;
use Throwable;

use function class_exists;

// Инициализация системы ролей
final class RolesInitializer
{
    // Инициализация системы ролей
    // ОТКЛЮЧЕНО: Системные роли и права больше не создаются автоматически
    // Root пользователь (userId = 0) имеет полный доступ без ролей
    // Администратор сам создает роли и права через админ-панель
    public static function initialize(): void
    {
        // Метод отключен - системные роли и права не создаются автоматически
        // Root пользователь (userId = 0) имеет полный доступ без проверки ролей
        // Администратор создает роли и права через админ-панель по необходимости
        return;
    }

    // Проверка существования таблицы
    private static function tableExists(PDO $db, string $tableName): bool
    {
        try {
            $stmt = $db->prepare('
                SELECT COUNT(*) as count
                FROM information_schema.tables
                WHERE table_schema = DATABASE()
                AND table_name = ?
            ');
            $stmt->execute([$tableName]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);

            return isset($result['count']) && (int)$result['count'] > 0;
        } catch (Throwable $e) {
            return false;
        }
    }

    // Инициализация базовых ролей
    private static function initializeRoles(PDO $db): void
    {
        $roles = [
            [
                'name' => 'Администратор',
                'slug' => 'admin',
                'description' => 'Полный доступ к системе',
                'is_system' => 1,
            ],
            [
                'name' => 'Редактор',
                'slug' => 'editor',
                'description' => 'Доступ к редактированию контента',
                'is_system' => 1,
            ],
            [
                'name' => 'Автор',
                'slug' => 'author',
                'description' => 'Доступ к созданию контента',
                'is_system' => 1,
            ],
            [
                'name' => 'Пользователь',
                'slug' => 'user',
                'description' => 'Базовый доступ',
                'is_system' => 1,
            ],
        ];

        foreach ($roles as $role) {
            $stmt = $db->prepare('
                SELECT id FROM roles WHERE slug = ?
            ');
            $stmt->execute([$role['slug']]);
            $existing = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$existing) {
                $stmt = $db->prepare('
                    INSERT INTO roles (name, slug, description, is_system)
                    VALUES (?, ?, ?, ?)
                ');
                $stmt->execute([
                    $role['name'],
                    $role['slug'],
                    $role['description'],
                    $role['is_system'],
                ]);
            }
        }
    }

    // Инициализация базовых разрешений
    private static function initializePermissions(PDO $db): void
    {
        if (!self::tableExists($db, 'permissions')) {
            return;
        }

        $permissions = [
            [
                'name' => 'Доступ к админ-панели',
                'slug' => 'admin.access',
                'description' => 'Доступ к административной панели',
                'category' => 'admin',
            ],
            [
                'name' => 'Управление пользователями',
                'slug' => 'admin.users.manage',
                'description' => 'Создание, редактирование и удаление пользователей',
                'category' => 'admin',
            ],
            [
                'name' => 'Управление ролями',
                'slug' => 'admin.roles.manage',
                'description' => 'Создание, редактирование и удаление ролей',
                'category' => 'admin',
            ],
            [
                'name' => 'Управление плагинами',
                'slug' => 'admin.plugins.manage',
                'description' => 'Установка, активация и деактивация плагинов',
                'category' => 'admin',
            ],
            [
                'name' => 'Управление темами',
                'slug' => 'admin.themes.manage',
                'description' => 'Установка и активация тем',
                'category' => 'admin',
            ],
            [
                'name' => 'Управление настройками',
                'slug' => 'admin.settings.manage',
                'description' => 'Изменение системных настроек',
                'category' => 'admin',
            ],
        ];

        foreach ($permissions as $permission) {
            $stmt = $db->prepare('
                SELECT id FROM permissions WHERE slug = ?
            ');
            $stmt->execute([$permission['slug']]);
            $existing = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$existing) {
                $stmt = $db->prepare('
                    INSERT INTO permissions (name, slug, description, category)
                    VALUES (?, ?, ?, ?)
                ');
                $stmt->execute([
                    $permission['name'],
                    $permission['slug'],
                    $permission['description'],
                    $permission['category'],
                ]);
            }
        }
    }
}
