<?php

declare(strict_types=1);

namespace Flowaxy\Bootstrap;

use Flowaxy\Core\System\Container\Container;
use Flowaxy\Core\System\PathResolver;
use RuntimeException;
use Throwable;

use function class_exists;
use function file_exists;
use function is_readable;
use function defined;

use const DIRECTORY_SEPARATOR;

// Ініціалізатор контейнера залежностей (DI Container)
final class ContainerInitializer
{
    // Ініціалізація контейнера залежностей
    public static function initialize(): Container
    {
        if (!class_exists(PathResolver::class)) {
            throw new RuntimeException('PathResolver not loaded');
        }

        $flowaxyDir = PathResolver::flowaxy();
        $engineDir = PathResolver::engine();
        $ds = defined('DS') ? DS : DIRECTORY_SEPARATOR;

        // Завантажуємо Container клас
        $containerInterfaceFile = $flowaxyDir . $ds . 'Core' . $ds . 'Contracts' . $ds . 'ContainerInterface.php';
        if (file_exists($containerInterfaceFile) && is_readable($containerInterfaceFile)) {
            require_once $containerInterfaceFile;
        }

        // Завантажуємо ServiceTags перед Container
        $serviceTagsFile = $flowaxyDir . $ds . 'Core' . $ds . 'System' . $ds . 'Container' . $ds . 'ServiceTags.php';
        if (file_exists($serviceTagsFile) && is_readable($serviceTagsFile)) {
            require_once $serviceTagsFile;
        }

        // Завантажуємо Container
        $containerFile = $flowaxyDir . $ds . 'Core' . $ds . 'System' . $ds . 'Container' . $ds . 'Container.php';
        if (!file_exists($containerFile) || !is_readable($containerFile)) {
            // Fallback для зворотної сумісності
            $containerFile = $engineDir . $ds . 'core' . $ds . 'system' . $ds . 'Container.php';
        }

        if (!file_exists($containerFile) || !is_readable($containerFile)) {
            throw new RuntimeException('Container file not found or not readable: ' . $containerFile);
        }

        require_once $containerFile;

        $containerClass = Container::class;
        if (!class_exists($containerClass)) {
            throw new RuntimeException('Container class not found: ' . $containerClass . '. File exists: ' . $containerFile);
        }

        try {
            $container = new $containerClass();
            $GLOBALS['engineContainer'] = $container;
            return $container;
        } catch (Throwable $e) {
            throw new RuntimeException('Failed to instantiate Container: ' . $e->getMessage() . '. Class: ' . $containerClass, 0, $e);
        }
    }
}
