<?php

declare(strict_types=1);

namespace Flowaxy\Bootstrap;

use Flowaxy\Core\System\PathResolver;
use Throwable;

use function class_exists;
use function defined;
use function define;
use function file_exists;
use function is_readable;
use function parse_ini_file;

use const DIRECTORY_SEPARATOR;

// Завантажувач конфігурації бази даних
final class ConfigLoader
{
    // Завантаження конфігурації БД з database.ini та визначення констант
    public static function loadDatabaseConfig(): void
    {
        // Перевіряємо, чи константи вже визначені
        if (defined('DB_HOST') && defined('DB_NAME')) {
            return;
        }

        if (!class_exists(PathResolver::class)) {
            return;
        }

        $ds = defined('DS') ? DS : DIRECTORY_SEPARATOR;
        // database.ini должен быть в storage/config/system/
        $databaseIniFile = PathResolver::storageConfig() . $ds . 'system' . $ds . 'database.ini';

        if (!file_exists($databaseIniFile) || !is_readable($databaseIniFile)) {
            // ВАЖЛИВО: database.ini НЕ створюємо автоматично.
            // Якщо файлу немає — БД вважається не налаштованою (Root може працювати без БД).
            return;
        }

        try {
            $config = parse_ini_file($databaseIniFile, true);
            if ($config === false || !isset($config['database'])) {
                return;
            }

            $db = $config['database'];
            // Глобальне правило:
            // - якщо status=false (або відсутній/порожній) — не визначаємо константи і не чіпаємо БД.
            $status = $db['status'] ?? false;
            $statusString = is_string($status) ? strtolower(trim($status)) : $status;
            $enabled = $statusString === true || $statusString === 1 || $statusString === '1' || $statusString === 'true' || $statusString === 'yes' || $statusString === 'on';
            if (!$enabled) {
                return;
            }

            $host = $db['host'] ?? '127.0.0.1';
            $port = isset($db['port']) ? (int)$db['port'] : 3306;
            $name = $db['name'] ?? '';
            $user = $db['user'] ?? 'root';
            $pass = $db['pass'] ?? '';
            $charset = $db['charset'] ?? 'utf8mb4';

            // Визначаємо константи БД, якщо вони ще не визначені
            if (!defined('DB_HOST')) {
                define('DB_HOST', $host . ':' . $port);
            }
            if (!defined('DB_NAME') && !empty($name)) {
                define('DB_NAME', $name);
            }
            if (!defined('DB_USER')) {
                define('DB_USER', $user);
            }
            if (!defined('DB_PASS')) {
                define('DB_PASS', $pass);
            }
            if (!defined('DB_CHARSET')) {
                define('DB_CHARSET', $charset);
            }
        } catch (Throwable) {
            // Ігноруємо помилки завантаження конфігурації БД (це нормально, якщо система ще не встановлена)
        }
    }
}
