<?php

/**
 * Default Theme
 *
 * @name: Default
 * @slug: default
 * @version: 1.0.0
 * @description: Тема по умолчанию для Flowaxy CMS
 * @author: Flowaxy Team
 * @author_url: https://flowaxy.com
 * @requires: 1.0.0
 * @tested: 1.0.0
 * @package: DefaultTheme
 */

declare(strict_types=1);

use Flowaxy\Support\Base\BaseTheme;

final class DefaultTheme extends BaseTheme
{
    /**
     * Список локаций меню, которые поддерживает тема
     * Ключ - идентификатор локации, значение - название
     *
     * @var array<string, string>
     */
    private array $menuLocations = [
        'header_left' => 'Header Left Menu',
        'header_center' => 'Header Center Menu',
        'footer' => 'Footer Menu',
    ];

    /**
     * Конфигурация страниц админки темы
     *
     * Этот массив автоматически регистрирует страницы админки для темы.
     * Чтобы добавить новую страницу, просто добавьте элемент в этот массив.
     *
     * Структура:
     * - 'menu' - настройки пункта меню (text, icon, title)
     * - 'callback' - имя функции страницы (должна быть определена в файле страницы)
     * - 'file' - имя файла страницы в content/admin/pages/
     * - 'enabled' - метод класса или callable для проверки доступности страницы
     *
     * Пример добавления новой страницы:
     * 'my-page' => [
     *     'menu' => ['text' => 'Моя страница', 'icon' => 'fas fa-star'],
     *     'callback' => 'admin_my_page',
     *     'file' => 'my-page.php',
     *     'enabled' => true, // или 'hasMyFeature' для проверки через метод класса
     * ],
     *
     * @var array<string, array{
     *     menu: array{text: string, icon: string, title?: string},
     *     callback: string,
     *     file: string,
     *     enabled: callable|bool|string
     * }>
     */
    private array $adminThemePages = [
        'theme-navigation' => [
            'menu' => [
                'text' => 'Навігація',
                'icon' => 'fas fa-bars',
                'title' => 'Налаштування меню навігації теми',
            ],
            'callback' => 'admin_theme_navigation_page',
            'file' => 'theme-navigation.php',
            'enabled' => 'hasMenus', // Метод класса для проверки доступности
        ],
        'widgets' => [
            'menu' => [
                'text' => 'Виджеты',
                'icon' => 'fas fa-puzzle-piece',
                'title' => 'Управление виджетами темы',
            ],
            'callback' => 'admin_widgets_page',
            'file' => 'widgets.php',
            'enabled' => 'hasWidgets', // Метод класса для проверки доступности
        ],
        'theme-customization' => [
            'menu' => [
                'text' => 'Кастоматизация',
                'icon' => 'fas fa-paint-brush',
                'title' => 'Кастоматизация темы',
            ],
            'callback' => 'admin_theme_customization_page',
            'file' => 'theme-customization.php',
            'enabled' => 'hasCustomization', // Метод класса для проверки доступности
        ],
    ];

    /**
     * Регистрация всех хуков темы (support + меню)
     */
    public function registerHooks(): void
    {
        // Поддержка возможностей темы
        if (function_exists('add_action') && function_exists('add_theme_support')) {
            add_action('theme_support', function () {
                // Виджеты (theme widgets API ядра Flowaxy)
                add_theme_support('widgets');

                // Поддержка меню (динамически из $this->menuLocations)
                add_theme_support('menus', array_keys($this->menuLocations));
            });
        }

        // Регистрация областей меню и сообщение об их наличии в админку
        if (function_exists('add_filter')) {
            // Регистрация областей меню темы (через action, если он доступен)
            if (function_exists('add_action')) {
                add_action('theme_init', function () {
                    if (function_exists('register_nav_menu')) {
                        // Регистрируем только те меню, которые есть в $this->menuLocations
                        foreach ($this->menuLocations as $location => $name) {
                            register_nav_menu($location, $name);
                        }
                    }
                });
            }

            // Тема сообщает в админку, какие меню она поддерживает
            // Данные берутся динамически из $this->menuLocations
            add_filter('theme_navigation_locations', function (array $locations): array {
                return array_merge($locations, $this->menuLocations);
            });

            // Тема сообщает в админку список виджетов
            add_filter('theme_widgets', function (array $widgets): array {
                return array_merge($widgets, $this->getThemeWidgets());
            });

            // Тема сообщает в админку настройки кастоматизации
            add_filter('theme_customization_settings', function (array $settings): array {
                return array_merge($settings, $this->getCustomizationSettings());
            });

            // Регистрация пунктов меню админки для страниц темы (через единый массив конфигурации)
            add_filter('admin_menu', function (array $menu): array {
                $themeSubmenu = [];

                // Проходим по всем страницам из конфигурации
                foreach ($this->adminThemePages as $pageId => $pageConfig) {
                    // Проверяем, доступна ли страница
                    $enabled = $pageConfig['enabled'] ?? true;
                    if (is_string($enabled) && method_exists($this, $enabled)) {
                        $enabled = $this->{$enabled}();
                    } elseif (is_callable($enabled)) {
                        $enabled = $enabled();
                    }

                    if ($enabled) {
                        $menuConfig = $pageConfig['menu'];
                        $themeSubmenu[] = [
                            'text' => $menuConfig['text'],
                            'title' => $menuConfig['title'] ?? $menuConfig['text'],
                            'icon' => $menuConfig['icon'],
                            'href' => admin_url($pageId),
                            'page' => $pageId,
                            'permission' => 'admin.access',
                        ];
                    }
                }

                // Если есть хотя бы один подпункт, добавляем родительскую категорию
                if (!empty($themeSubmenu)) {
                    $menu[] = [
                        'text' => 'Тема',
                        'title' => 'Настройки темы',
                        'icon' => 'fas fa-palette',
                        'href' => '#',
                        'page' => 'theme',
                        'permission' => 'admin.access',
                        'order' => 32,
                        'submenu' => $themeSubmenu,
                    ];
                }

                return $menu;
            });
        }

        // Регистрация маршрутов админки для страниц темы (через единый массив конфигурации)
        if (function_exists('add_action')) {
            add_action('admin_register_routes', function ($router): void {
                if (!function_exists('path_content_admin') || !$router || !method_exists($router, 'add')) {
                    return;
                }

                $pagesDir = path_content_admin() . DIRECTORY_SEPARATOR . 'pages';

                // Проходим по всем страницам из конфигурации
                foreach ($this->adminThemePages as $pageId => $pageConfig) {
                    // Проверяем, доступна ли страница
                    $enabled = $pageConfig['enabled'] ?? true;
                    if (is_string($enabled) && method_exists($this, $enabled)) {
                        $enabled = $this->{$enabled}();
                    } elseif (is_callable($enabled)) {
                        $enabled = $enabled();
                    }

                    if ($enabled) {
                        $pageFile = $pagesDir . DIRECTORY_SEPARATOR . $pageConfig['file'];
                        if (file_exists($pageFile) && is_readable($pageFile)) {
                            require_once $pageFile;
                        }

                        $router->add(['GET', 'POST'], $pageId, $pageConfig['callback'], ['middleware' => ['admin.auth']]);
                    }
                }
            });
        }
    }

    /**
     * Получение списка виджетов темы
     *
     * @return array<string, array{name: string, description: string}>
     */
    private function getThemeWidgets(): array
    {
        $widgetsDir = $this->themeDir . 'widgets';
        $widgets = [];

        if (!is_dir($widgetsDir)) {
            return $widgets;
        }

        $items = scandir($widgetsDir);
        foreach ($items as $item) {
            if ($item === '.' || $item === '..' || $item === 'README.md') {
                continue;
            }

            $widgetPath = $widgetsDir . DS . $item;
            if (is_dir($widgetPath)) {
                $widgetFile = $widgetPath . DS . 'widget.php';
                if (file_exists($widgetFile)) {
                    $widgets[$item] = [
                        'name' => ucfirst(str_replace(['-', '_'], ' ', $item)),
                        'description' => 'Виджет темы: ' . $item,
                    ];
                }
            }
        }

        return $widgets;
    }

    /**
     * Получение настроек кастоматизации из файла customization.php
     *
     * @return array<string, mixed>
     */
    private function getCustomizationSettings(): array
    {
        $customizationFile = $this->themeDir . 'customization.php';

        if (!file_exists($customizationFile)) {
            return [];
        }

        try {
            $settings = require $customizationFile;
            return is_array($settings) ? $settings : [];
        } catch (\Throwable $e) {
            // В случае ошибки возвращаем пустой массив
            return [];
        }
    }

    /**
     * Проверка, поддерживает ли тема меню
     *
     * @return bool
     */
    private function hasMenus(): bool
    {
        return !empty($this->menuLocations);
    }

    /**
     * Проверка, поддерживает ли тема виджеты
     *
     * @return bool
     */
    private function hasWidgets(): bool
    {
        $widgetsDir = $this->themeDir . 'widgets';
        return is_dir($widgetsDir) && !empty($this->getThemeWidgets());
    }

    /**
     * Проверка, имеет ли тема файл кастоматизации
     *
     * @return bool
     */
    private function hasCustomization(): bool
    {
        $customizationFile = $this->themeDir . 'customization.php';
        return file_exists($customizationFile) && !empty($this->getCustomizationSettings());
    }

    public function registerRoutes(): void {}

    public function renderTheme(): void
    {
        $this->render('default');
    }

    public function activate(): void
    {
        parent::activate();
    }

    public function deactivate(): void
    {
        parent::deactivate();
    }
}
