# Social Networks Plugin

Плагин для управления социальными сетями в Flowaxy CMS. Предоставляет данные через хуки для использования в темах.

## Описание

Плагин позволяет управлять списком социальных сетей через админ-панель и предоставляет данные через хуки для отображения в любой теме. Поддерживает включение/отключение отдельных сетей, настройку иконок Font Awesome и порядок отображения.

## Установка

1. Перейдите в админ-панель → **Плагины**
2. Найдите плагин **"Social Networks"** в списке
3. Нажмите **"Установить"** (если плагин не установлен)
4. Нажмите **"Активировать"**

После активации плагин автоматически создаст файл конфигурации `config/networks.ini` с примерами социальных сетей.

## Настройка

### Управление социальными сетями

1. Перейдите в админ-панель → **Социальные сети**
2. Для каждой сети можно настроить:

    - **ID** — уникальный идентификатор (например: `tiktok`, `instagram`)
    - **Название** — отображаемое имя (например: `TikTok`, `Instagram`)
    - **URL** — ссылка на страницу в социальной сети
    - **Иконка** — класс Font Awesome (например: `fab fa-tiktok`, `fab fa-instagram`)
    - **Порядок** — порядок отображения (чем меньше число, тем выше в списке)
    - **Статус** — включена/выключена сеть

3. Нажмите **"Добавить"** для добавления новой сети
4. Нажмите **"Сохранить"** для сохранения изменений

### Настройки плагина

1. В админ-панели → **Социальные сети** нажмите кнопку **"Настройки"**
2. Доступные настройки:
    - **Статус плагина** — активирован/деактивирован
        - При деактивации социальные сети не отображаются в теме

## Использование в теме

Плагин предоставляет данные через хуки. Вы можете использовать их в любой теме двумя способами:

### Способ 1: Использование готового компонента (если есть)

Если в вашей теме есть компонент `social-networks`, используйте его:

```php
<?php theme_component('social-networks'); ?>
```

**Примечание:** Компонент должен быть создан в теме по пути:

```
content/themes/{theme-name}/components/social-networks/component.php
```

### Способ 2: Использование хуков напрямую (рекомендуется)

Если компонента нет, используйте хуки плагина напрямую в шаблонах темы:

#### Получение списка социальных сетей

```php
<?php
$networks = [];
if (class_exists('\Flowaxy\Core\Hooks\Filter')) {
    $networks = \Flowaxy\Core\Hooks\Filter::apply('theme_social_networks', $networks, []);
} elseif (function_exists('apply_filters')) {
    $networks = apply_filters('theme_social_networks', $networks);
}

// Проверяем настройки плагина
$settings = [];
if (class_exists('\Flowaxy\Infrastructure\Config\PluginSettingsManager')) {
    $settings = \Flowaxy\Infrastructure\Config\PluginSettingsManager::getSettings('social-networks');
}
$status = $settings['status'] ?? 'activated';

// Отображаем только если плагин активирован и есть сети
if ($status === 'activated' && !empty($networks)) {
    echo '<div class="social-share">';
    foreach ($networks as $network) {
        $url = $network['url'] ?? '#';
        $enabled = $network['enabled'] ?? true;

        // Пропускаем отключенные сети и невалидные URL
        if (!$enabled || empty($url) || $url === '#') {
            continue;
        }

        echo '<a href="' . htmlspecialchars($url, ENT_QUOTES, 'UTF-8') . '"';
        echo ' class="share-icon ' . htmlspecialchars($network['id'] ?? '', ENT_QUOTES, 'UTF-8') . '"';
        echo ' title="' . htmlspecialchars($network['name'] ?? '', ENT_QUOTES, 'UTF-8') . '"';
        echo ' target="_blank" rel="noopener noreferrer">';
        echo '<i class="' . htmlspecialchars($network['icon'] ?? 'fas fa-link', ENT_QUOTES, 'UTF-8') . '"></i>';
        echo '</a>';
    }
    echo '</div>';
}
?>
```

#### Структура данных

Каждая социальная сеть в массиве содержит:

```php
[
    'id' => 'tiktok',              // Уникальный ID
    'name' => 'TikTok',            // Название
    'url' => 'https://...',        // URL страницы
    'icon' => 'fab fa-tiktok',     // Класс Font Awesome
    'order' => 1,                  // Порядок сортировки
    'enabled' => true              // Включена/выключена
]
```

## Создание компонента в теме

Если вы хотите создать компонент для использования в теме:

1. Создайте директорию:

    ```
    content/themes/{theme-name}/components/social-networks/
    ```

2. Создайте файл `component.php`:

```php
<?php
$networks = [];
if (class_exists('\Flowaxy\Core\Hooks\Filter')) {
    $networks = \Flowaxy\Core\Hooks\Filter::apply('theme_social_networks', $networks, []);
}

$settings = [];
if (class_exists('\Flowaxy\Infrastructure\Config\PluginSettingsManager')) {
    $settings = \Flowaxy\Infrastructure\Config\PluginSettingsManager::getSettings('social-networks');
}

$status = $settings['status'] ?? 'activated';
$validNetworks = [];

if (!empty($networks) && is_array($networks)) {
    foreach ($networks as $network) {
        $url = $network['url'] ?? '#';
        $enabled = $network['enabled'] ?? true;
        if ($enabled && !empty($url) && $url !== '#' && filter_var($url, FILTER_VALIDATE_URL)) {
            $validNetworks[] = $network;
        }
    }
}

if ($status !== 'activated' || empty($validNetworks)) {
    return;
}
?>

<div class="social-networks-component">
    <div class="social-share">
        <?php foreach ($validNetworks as $network): ?>
            <a href="<?php echo htmlspecialchars($network['url'] ?? '#', ENT_QUOTES, 'UTF-8'); ?>"
               class="share-icon <?php echo htmlspecialchars($network['id'] ?? '', ENT_QUOTES, 'UTF-8'); ?>"
               title="<?php echo htmlspecialchars($network['name'] ?? '', ENT_QUOTES, 'UTF-8'); ?>"
               target="_blank"
               rel="noopener noreferrer">
                <i class="<?php echo htmlspecialchars($network['icon'] ?? 'fas fa-link', ENT_QUOTES, 'UTF-8'); ?>"></i>
            </a>
        <?php endforeach; ?>
    </div>
</div>
```

3. (Опционально) Создайте файл `component.css` для стилизации:

```css
.social-networks-component .social-share {
    display: flex;
    gap: 6px;
}

.social-networks-component .share-icon {
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    background-color: #f5f5f5;
    border: 1px solid #e0e0e0;
    border-radius: 5px;
    color: #333333;
    font-size: 12px;
    font-weight: 600;
    text-decoration: none;
    transition: background-color 0.2s, border-color 0.2s;
}

.social-networks-component .share-icon:hover {
    background-color: #e9ecef;
    border-color: #4caf50;
}
```

## Хуки плагина

Плагин предоставляет следующие хуки:

### `theme_social_networks`

Возвращает массив социальных сетей. Используется через:

```php
$networks = Filter::apply('theme_social_networks', []);
```

### `theme_social_networks_settings`

Возвращает настройки плагина. Используется через:

```php
$settings = Filter::apply('theme_social_networks_settings', []);
// Результат: ['status' => 'activated' | 'deactivated']
```

## Структура файлов плагина

```
content/plugins/social-networks/
├── Plugin.php                    # Основной класс плагина
├── README.md                     # Документация
├── admin/
│   └── pages/
│       └── social-networks.php  # Страница управления в админке
└── config/
    └── networks.ini              # Конфигурация социальных сетей
```

## Конфигурационные файлы

### `config/networks.ini`

Файл хранит список социальных сетей (создается автоматически):

```ini
[networks]

[networks.tiktok]
name = "TikTok"
url = "https://www.tiktok.com/@yourpage"
icon = "fab fa-tiktok"
order = 1
enabled = 1
```

### `storage/config/plugins/social-networks.ini`

Файл хранит настройки плагина:

```ini
[settings]
status = activated
```

## Что поддерживается

-   ✅ Добавление/редактирование/удаление социальных сетей
-   ✅ Включение/отключение отдельных сетей
-   ✅ Настройка порядка отображения
-   ✅ Использование иконок Font Awesome
-   ✅ Активация/деактивация плагина
-   ✅ Валидация URL
-   ✅ Автоматическая сортировка по порядку
-   ✅ Работа через хуки (не требует компонента в теме)

## Иконки Font Awesome

Плагин поддерживает все иконки Font Awesome. Примеры:

-   `fab fa-tiktok` — TikTok
-   `fab fa-instagram` — Instagram
-   `fab fa-telegram` — Telegram
-   `fab fa-whatsapp` — WhatsApp
-   `fab fa-facebook` — Facebook
-   `fab fa-vk` — VKontakte
-   `fab fa-twitter` — Twitter/X
-   `fab fa-youtube` — YouTube
-   И другие...

## Требования

-   Flowaxy CMS 0.0.2 PRE+
-   PHP 8.4+
-   Font Awesome (для отображения иконок, должен быть подключен в теме)

## Автор

Flowaxy Team
https://flowaxy.com

## Версия: 1.0.0
