/**
 * Flowaxy Admin Panel - Themes Page JavaScript
 * Управление темами: активация, компиляция, удаление
 */

(function() {
    'use strict';

    const csrfToken = document.querySelector('input[name="csrf_token"]')?.value || '';

    document.addEventListener('DOMContentLoaded', function() {
        initThemeActivation();
        initThemeDeletion();
    });

    function initThemeDeletion() {
        document.querySelectorAll('.delete-theme-btn').forEach(function(btn) {
            btn.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();

                const themeSlug = this.dataset.themeSlug;
                if (!themeSlug) {
                    alert('Помилка: не вказано slug теми');
                    return false;
                }

                deleteTheme(themeSlug);
                return false;
            });
        });
    }

    function initThemeActivation() {
        document.querySelectorAll('.theme-activate-form').forEach(function(form) {
            form.addEventListener('submit', function(e) {
                e.preventDefault();

                const themeSlug = this.dataset.themeSlug;
                const hasScss = this.dataset.hasScss === '1';
                const btn = this.querySelector('.theme-activate-btn');

                if (!btn) {
                    if (window.DEBUG_MODE) {
                        console.error('Кнопка активации не найдена');
                    }
                    return;
                }

                const btnSpinner = btn.querySelector('.btn-spinner');
                const originalText = btn.textContent.trim();

                btn.disabled = true;
                btn.classList.add(hasScss ? 'compiling' : 'activating');
                if (btnSpinner) {
                    btnSpinner.style.display = 'inline-flex';
                }

                if (hasScss) {
                    updateButtonText(btn, 'Компілюється...');
                    compileAndActivateTheme(themeSlug, btn, originalText, btnSpinner);
                } else {
                    updateButtonText(btn, 'Активується...');
                    activateTheme(themeSlug, btn, originalText, btnSpinner);
                }
            });
        });
    }

    function updateButtonText(btn, newText) {
        const icon = btn.querySelector('i');
        if (icon) {
            const textNode = Array.from(btn.childNodes).find(node =>
                node.nodeType === Node.TEXT_NODE && node.textContent.trim()
            );
            if (textNode) {
                textNode.textContent = ' ' + newText;
            } else {
                btn.appendChild(document.createTextNode(' ' + newText));
            }
        } else {
            btn.textContent = newText;
        }
    }

    function compileAndActivateTheme(themeSlug, btn, originalText, btnSpinner) {
        const formData = new FormData();
        formData.append('action', 'activate_theme');
        formData.append('theme_slug', themeSlug);
        formData.append('csrf_token', csrfToken);

        fetch('', {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                updateButtonText(btn, 'Активовано!');
                btn.classList.remove('compiling');
                btn.classList.add('activating');
                setTimeout(function() {
                    window.location.reload();
                }, 1000);
            } else {
                btn.disabled = false;
                btn.classList.remove('compiling', 'activating');
                if (btnSpinner) {
                    btnSpinner.style.display = 'none';
                }
                updateButtonText(btn, originalText);
                alert('Помилка: ' + (data.error || 'Невідома помилка'));
            }
        })
        .catch(error => {
            btn.disabled = false;
            btn.classList.remove('compiling', 'activating');
            if (btnSpinner) {
                btnSpinner.style.display = 'none';
            }
            updateButtonText(btn, originalText);
            alert('Помилка підключення до сервера');
            if (window.DEBUG_MODE) {
                console.error('Error:', error);
            }
        });
    }

    function activateTheme(themeSlug, btn, originalText, btnSpinner) {
        const formData = new FormData();
        formData.append('action', 'activate_theme');
        formData.append('theme_slug', themeSlug);
        formData.append('csrf_token', csrfToken);

        fetch('', {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => {
            const contentType = response.headers.get('content-type');
            if (contentType && contentType.includes('application/json')) {
                return response.json();
            } else {
                throw new Error('Неверный формат ответа от сервера');
            }
        })
        .then(data => {
            if (data.success) {
                updateButtonText(btn, 'Активовано!');
                btn.classList.remove('activating');
                btn.classList.add('success');
                setTimeout(function() {
                    window.location.reload();
                }, 1000);
            } else {
                btn.disabled = false;
                btn.classList.remove('activating');
                if (btnSpinner) {
                    btnSpinner.style.display = 'none';
                }
                updateButtonText(btn, originalText);
                alert('Помилка: ' + (data.error || 'Невідома помилка'));
                if (window.DEBUG_MODE) {
                    console.error('Activation error:', data);
                }
            }
        })
        .catch(error => {
            btn.disabled = false;
            btn.classList.remove('activating');
            if (btnSpinner) {
                btnSpinner.style.display = 'none';
            }
            updateButtonText(btn, originalText);
            alert('Помилка підключення до сервера');
            if (window.DEBUG_MODE) {
                console.error('Error:', error);
            }
        });
    }

    function deleteTheme(slug) {
        if (!slug) {
            alert('Помилка: не вказано slug теми');
            return false;
        }

        if (confirm('Ви впевнені, що хочете видалити цю тему?\n\nБудуть видалені:\n- Всі файли теми\n- Всі налаштування теми з бази даних (theme_settings)\n\nЦю дію неможливо скасувати!')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = window.location.href;

            const csrfInput = document.getElementById('csrf_token') || document.querySelector('input[name="csrf_token"]');
            const csrfTokenValue = csrfInput ? csrfInput.value : '';

            if (!csrfTokenValue) {
                alert('Помилка: не знайдено CSRF токен');
                if (window.DEBUG_MODE) {
                    console.error('CSRF token not found');
                }
                return false;
            }

            form.innerHTML = `
                <input type="hidden" name="csrf_token" value="${csrfTokenValue}">
                <input type="hidden" name="action" value="delete_theme">
                <input type="hidden" name="theme_slug" value="${slug}">
            `;

            document.body.appendChild(form);
            if (window.DEBUG_MODE) {
                console.log('Submitting delete theme form for:', slug);
            }
            form.submit();
        }

        return false;
    }
})();
