/**
 * Flowaxy Admin Panel - Site Settings Page JavaScript
 * Управление настройками сайта: часовые пояса, логирование
 */

(function() {
    'use strict';

    document.addEventListener('DOMContentLoaded', function() {
        initLoggingRotation();
        initTimezoneSelector();
        initCurrentTimeUpdater();
    });

    // Инициализация управления ротацией логов
    function initLoggingRotation() {
        const rotationType = document.getElementById('loggingRotationType');
        const rotationTimeGroup = document.getElementById('rotationTimeGroup');

        if (rotationType && rotationTimeGroup) {
            rotationType.addEventListener('change', function() {
                if (this.value === 'time' || this.value === 'both') {
                    rotationTimeGroup.style.display = 'block';
                } else {
                    rotationTimeGroup.style.display = 'none';
                }
            });
        }
    }

    // Инициализация селектора часовых поясов
    function initTimezoneSelector() {
        const timezoneSelect = document.getElementById('timezone');
        const timezoneSearch = document.getElementById('timezoneSearch');
        const timezoneDisplay = document.getElementById('timezoneDisplay');
        const timezoneDropdown = document.getElementById('timezoneDropdown');
        const timezoneArrow = document.querySelector('.timezone-arrow');
        const timezoneWrapper = document.querySelector('.timezone-select-wrapper');

        if (!timezoneSelect || !timezoneSearch || !timezoneDisplay || !timezoneDropdown) {
            return;
        }

        // Функция для обновления отображения выбранного значения
        function updateDisplay() {
            const selectedOption = Array.from(timezoneSelect.options).find(function(opt) {
                return opt.selected === true;
            });

            if (selectedOption) {
                const displayText = selectedOption.getAttribute('data-text') || selectedOption.text;
                timezoneDisplay.value = displayText;
            } else {
                const currentValue = timezoneSelect.value;
                if (currentValue && currentValue !== '') {
                    const foundOption = Array.from(timezoneSelect.options).find(function(opt) {
                        return opt.value === currentValue;
                    });
                    if (foundOption) {
                        foundOption.selected = true;
                        timezoneDisplay.value = foundOption.getAttribute('data-text') || foundOption.text;
                    } else {
                        if (timezoneDisplay.value === '') {
                            timezoneDisplay.value = currentValue;
                        }
                    }
                } else {
                    timezoneDisplay.value = 'Не вибрано';
                    timezoneSelect.value = '';
                }
            }
        }

        // Переконуємося, що поле відображення видиме, а пошукове приховане
        timezoneDisplay.style.display = 'block';
        timezoneSearch.style.display = 'none';

        // Оновлюємо відображення при завантаженні сторінки
        const selectedOptionOnLoad = Array.from(timezoneSelect.options).find(function(opt) {
            return opt.selected === true || opt.hasAttribute('selected');
        });

        if (selectedOptionOnLoad) {
            timezoneSelect.value = selectedOptionOnLoad.value;
            const displayText = selectedOptionOnLoad.getAttribute('data-text') || selectedOptionOnLoad.text;
            if (displayText && timezoneDisplay.value !== displayText) {
                timezoneDisplay.value = displayText;
            }
        } else {
            const currentDisplayValue = timezoneDisplay.value;
            if (currentDisplayValue) {
                const matchingOption = Array.from(timezoneSelect.options).find(function(opt) {
                    const optionText = opt.getAttribute('data-text') || opt.text;
                    return optionText === currentDisplayValue;
                });
                if (matchingOption) {
                    timezoneSelect.value = matchingOption.value;
                    matchingOption.selected = true;
                }
            } else {
                updateDisplay();
            }
        }

        // Фінальна перевірка
        const finalSelectedOption = Array.from(timezoneSelect.options).find(function(opt) {
            return opt.value === timezoneSelect.value;
        });
        if (finalSelectedOption) {
            const expectedText = finalSelectedOption.getAttribute('data-text') || finalSelectedOption.text;
            if (timezoneDisplay.value !== expectedText) {
                timezoneDisplay.value = expectedText;
            }
        } else if (!timezoneSelect.value || timezoneSelect.value === '') {
            if (timezoneDisplay.value === '' || !timezoneDisplay.value) {
                timezoneDisplay.value = 'Не вибрано';
            }
        }

        // Функція для фільтрації опцій
        function filterTimezones(searchTerm) {
            const options = timezoneSelect.options;
            const filteredOptions = [];

            for (let i = 0; i < options.length; i++) {
                const option = options[i];
                const text = option.getAttribute('data-text') || option.text;
                const value = option.value.toLowerCase();
                const searchLower = searchTerm.toLowerCase();

                if (text.toLowerCase().includes(searchLower) || value.includes(searchLower)) {
                    filteredOptions.push({
                        value: option.value,
                        text: text,
                        selected: option.selected
                    });
                }
            }

            filteredOptions.sort(function(a, b) {
                return a.text.localeCompare(b.text);
            });

            return filteredOptions;
        }

        // Функція для відображення випадаючого списку
        function showDropdown(options) {
            if (options.length === 0) {
                timezoneDropdown.innerHTML = '<div class="timezone-dropdown-item text-muted">Нічого не знайдено</div>';
            } else {
                const maxItems = options.length > 15 ? 15 : options.length;
                timezoneDropdown.innerHTML = options.slice(0, maxItems).map(function(opt) {
                    const activeClass = opt.selected ? 'active' : '';
                    return '<div class="timezone-dropdown-item ' + activeClass + '" data-value="' +
                           opt.value + '">' + opt.text + '</div>';
                }).join('');

                if (options.length > maxItems) {
                    timezoneDropdown.innerHTML += '<div class="timezone-dropdown-item text-muted" style="font-size: 0.75rem; padding: 0.25rem 0.75rem;">Показано ' + maxItems + ' з ' + options.length + '. Уточніть пошук для більшої кількості результатів.</div>';
                }
            }

            // Додаємо обробники подій для елементів списку
            timezoneDropdown.querySelectorAll('.timezone-dropdown-item').forEach(function(item) {
                item.addEventListener('click', function() {
                    const value = this.getAttribute('data-value');
                    const text = this.textContent.trim();

                    if (this.classList.contains('text-muted')) {
                        return;
                    }

                    timezoneSelect.value = value || '';

                    Array.from(timezoneSelect.options).forEach(function(opt) {
                        opt.selected = (opt.value === value);
                    });

                    if (value === '' || !value) {
                        timezoneDisplay.value = 'Не вибрано';
                        timezoneSelect.value = '';
                    } else {
                        if (typeof updateDisplay === 'function') {
                            updateDisplay();
                        } else {
                            timezoneDisplay.value = text;
                        }
                    }

                    timezoneDropdown.style.display = 'none';
                    timezoneSearch.value = '';
                    timezoneSearch.style.display = 'none';
                    timezoneDisplay.style.display = 'block';
                    timezoneSearch.blur();
                    if (timezoneArrow) {
                        timezoneArrow.classList.remove('rotate');
                    }

                    timezoneDisplay.style.backgroundColor = '#e7f3ff';
                    setTimeout(function() {
                        timezoneDisplay.style.backgroundColor = '';
                    }, 300);

                    if (typeof window.updateCurrentTime === 'function') {
                        const timezoneContainer = document.getElementById('timezone-current-time');
                        if (timezoneContainer && value) {
                            timezoneContainer.setAttribute('data-timezone', value);
                            window.updateCurrentTime();
                        }
                    }
                });
            });
        }

        // Обробка пошуку
        let searchTimeout;
        timezoneSearch.addEventListener('input', function() {
            clearTimeout(searchTimeout);
            const searchTerm = this.value.trim();

            searchTimeout = setTimeout(function() {
                if (searchTerm.length === 0) {
                    const options = [];
                    for (let i = 0; i < Math.min(15, timezoneSelect.options.length); i++) {
                        const option = timezoneSelect.options[i];
                        options.push({
                            value: option.value,
                            text: option.getAttribute('data-text') || option.text,
                            selected: option.selected
                        });
                    }
                    if (options.length > 0) {
                        showDropdown(options);
                        timezoneDropdown.style.display = 'block';
                    } else {
                        timezoneDropdown.style.display = 'none';
                    }
                } else {
                    const filtered = filterTimezones(searchTerm);
                    showDropdown(filtered);
                    timezoneDropdown.style.display = 'block';
                }
                if (timezoneArrow) {
                    timezoneArrow.classList.add('rotate');
                }
            }, 200);
        });

        // Показ/приховування випадаючого списку при кліку на поле відображення
        timezoneDisplay.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();

            timezoneSearch.style.display = 'block';
            timezoneSearch.value = '';
            timezoneSearch.focus();

            const options = [];
            for (let i = 0; i < Math.min(15, timezoneSelect.options.length); i++) {
                const option = timezoneSelect.options[i];
                options.push({
                    value: option.value,
                    text: option.getAttribute('data-text') || option.text,
                    selected: option.selected
                });
            }
            showDropdown(options);

            timezoneDropdown.style.display = 'block';
            if (timezoneArrow) {
                timezoneArrow.classList.add('rotate');
            }
        });

        // Закриття випадаючого списку при кліку поза ним
        if (timezoneWrapper) {
            document.addEventListener('click', function(e) {
                if (!timezoneWrapper.contains(e.target)) {
                    timezoneDropdown.style.display = 'none';
                    timezoneSearch.value = '';
                    timezoneSearch.style.display = 'none';
                    timezoneDisplay.style.display = 'block';
                    if (timezoneArrow) {
                        timezoneArrow.classList.remove('rotate');
                    }
                }
            });
        }

        // Обробка клавіатури для навігації
        timezoneSearch.addEventListener('keydown', function(e) {
            const items = timezoneDropdown.querySelectorAll('.timezone-dropdown-item');
            const activeItem = timezoneDropdown.querySelector('.timezone-dropdown-item.active');

            if (e.key === 'ArrowDown') {
                e.preventDefault();
                if (activeItem) {
                    const next = activeItem.nextElementSibling;
                    if (next) {
                        activeItem.classList.remove('active');
                        next.classList.add('active');
                        next.scrollIntoView({ block: 'nearest' });
                    }
                } else if (items.length > 0) {
                    items[0].classList.add('active');
                }
            } else if (e.key === 'ArrowUp') {
                e.preventDefault();
                if (activeItem) {
                    const prev = activeItem.previousElementSibling;
                    if (prev) {
                        activeItem.classList.remove('active');
                        prev.classList.add('active');
                        prev.scrollIntoView({ block: 'nearest' });
                    }
                }
            } else if (e.key === 'Enter') {
                e.preventDefault();
                if (activeItem) {
                    activeItem.click();
                }
            } else if (e.key === 'Escape') {
                timezoneDropdown.style.display = 'none';
                timezoneSearch.value = '';
                timezoneSearch.style.display = 'none';
                timezoneDisplay.style.display = 'block';
                if (timezoneArrow) {
                    timezoneArrow.classList.remove('rotate');
                }
            }
        });
    }

    // Инициализация обновления текущего времени
    function initCurrentTimeUpdater() {
        // Функція для оновлення поточного часу
        window.updateCurrentTime = function() {
            const timeDisplay = document.getElementById('current-time-display');
            const timezoneContainer = document.getElementById('timezone-current-time');

            if (!timeDisplay || !timezoneContainer) {
                return;
            }

            const timezone = timezoneContainer.getAttribute('data-timezone') || 'UTC';

            try {
                const now = new Date();
                const options = {
                    timeZone: timezone,
                    year: 'numeric',
                    month: '2-digit',
                    day: '2-digit',
                    hour: '2-digit',
                    minute: '2-digit',
                    second: '2-digit',
                    hour12: false
                };

                const formatter = new Intl.DateTimeFormat('uk-UA', options);
                const parts = formatter.formatToParts(now);

                const day = parts.find(p => p.type === 'day').value;
                const month = parts.find(p => p.type === 'month').value;
                const year = parts.find(p => p.type === 'year').value;
                const hour = parts.find(p => p.type === 'hour').value;
                const minute = parts.find(p => p.type === 'minute').value;
                const second = parts.find(p => p.type === 'second').value;

                const formattedTime = day + '.' + month + '.' + year + ' ' + hour + ':' + minute + ':' + second;
                timeDisplay.textContent = formattedTime;
            } catch (e) {
                const now = new Date();
                const day = String(now.getDate()).padStart(2, '0');
                const month = String(now.getMonth() + 1).padStart(2, '0');
                const year = now.getFullYear();
                const hours = String(now.getHours()).padStart(2, '0');
                const minutes = String(now.getMinutes()).padStart(2, '0');
                const seconds = String(now.getSeconds()).padStart(2, '0');
                timeDisplay.textContent = day + '.' + month + '.' + year + ' ' + hours + ':' + minutes + ':' + seconds;
            }
        };

        // Оновлюємо час кожну секунду
        setInterval(window.updateCurrentTime, 1000);

        // Оновлюємо час при зміні часового поясу через select
        const timezoneSelectForTime = document.getElementById('timezone');
        if (timezoneSelectForTime) {
            timezoneSelectForTime.addEventListener('change', function() {
                const timezoneContainer = document.getElementById('timezone-current-time');
                if (timezoneContainer && this.value) {
                    timezoneContainer.setAttribute('data-timezone', this.value);
                    window.updateCurrentTime();
                }
            });
        }

        // Первичное обновление времени
        window.updateCurrentTime();
    }
})();
