<?php
/**
 * Страница управления пользователями (функция вместо класса)
 */

/**
 * Страница управления пользователями
 *
 * Примечание: Авторизация обрабатывается автоматически через middleware 'admin.auth'
 * Разработчикам не нужно проверять авторизацию вручную - фреймворк делает это автоматически
 */
function admin_users_page() {
    // Проверка подключения к БД
    if (!function_exists('is_database_available') || !is_database_available()) {
        $layout = [
            'title' => 'Користувачі - Flowaxy CMS',
            'content' => render_database_required_message('Користувачі'),
            'pageHeaderIcon' => 'fas fa-users',
            'pageBreadcrumbs' => [
                ['title' => 'Головна', 'url' => admin_url('dashboard')],
                ['title' => 'Налаштування', 'url' => admin_url('settings')],
                ['title' => 'Користувачі'],
            ],
        ];
        return render_admin_layout($layout);
    }

    // Проверка прав доступа (только первый пользователь или с правами admin.access)
    $session = null;
    if (function_exists('sessionManager')) {
        $session = sessionManager();
    }
    $userId = $session ? (int)($session->get('admin_user_id') ?? 0) : 0;

    $hasAccess = ($userId === 0 || $userId === 1) || (function_exists('current_user_can') && current_user_can('admin.access'));
    if (!$hasAccess) {
        if (function_exists('admin_redirect')) {
            admin_redirect('dashboard');
        }
        return;
    }

    // Обработка POST запросов
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        match ($_POST['action']) {
            'create_user' => handle_create_user(),
            'update_user' => handle_update_user(),
            'delete_user' => handle_delete_user(),
            'change_password' => handle_change_password(),
            default => null,
        };
    }

    // Получаем данные через хуки
    $data = get_content_data('users_data', [
        'users' => get_users(),
        'roles' => get_all_roles(),
    ]);

    // Рендерим страницу
    $layout = [
        'title' => 'Користувачі - Flowaxy CMS',
        'content' => render_users_content($data),
        'pageHeaderIcon' => 'fas fa-users',
        'pageBreadcrumbs' => [
            ['title' => 'Головна', 'url' => admin_url('dashboard'), 'page' => 'dashboard'],
            ['title' => 'Налаштування', 'url' => admin_url('settings'), 'page' => 'settings'],
            ['title' => 'Користувачі', 'page' => 'users'],
        ],
    ];

    return render_admin_layout($layout);
}

/**
 * Рендеринг контента страницы пользователей
 */
function render_users_content($data) {
    ob_start();
    $templatePath = __DIR__ . DS . 'users' . DS . 'template.php';
    if (file_exists($templatePath)) {
        extract($data);
        include $templatePath;
    }
    return ob_get_clean();
}

/**
 * Получение всех пользователей
 */
function get_users() {
    $query = '
        SELECT id, username, email, is_active, last_activity, role_ids, created_at
        FROM users
        ORDER BY username
    ';

    $users = db_fetch_all($query);

    foreach ($users as &$user) {
        $user['is_active'] = (bool)$user['is_active'];
        $user['role_ids'] = !empty($user['role_ids'])
            ? json_decode($user['role_ids'], true) ?? []
            : [];

        // Получаем роли пользователя
        if (!empty($user['role_ids'])) {
            $userRoles = [];
            foreach ($user['role_ids'] as $roleId) {
                $role = db_fetch_one('SELECT name, slug FROM roles WHERE id = ?', [$roleId]);
                if ($role) {
                    $userRoles[] = $role;
                }
            }
            $user['roles'] = $userRoles;
        } else {
            $user['roles'] = [];
        }
    }

    return $users;
}

/**
 * Получение всех ролей
 */
function get_all_roles() {
    $query = '
        SELECT id, name, slug, description
        FROM roles
        ORDER BY name
    ';
    return db_fetch_all($query);
}

/**
 * Обработка создания пользователя
 */
function handle_create_user() {
    // Проверка CSRF будет добавлена позже
    $session = function_exists('sessionManager') ? sessionManager() : null;
    $userId = $session ? (int)($session->get('admin_user_id') ?? 0) : 0;

    $hasAccess = ($userId === 0 || $userId === 1) || (function_exists('current_user_can') && current_user_can('admin.access'));
    if (!$hasAccess) {
        set_flash_message('У вас немає прав на створення користувачів', 'danger');
        return;
    }

    $username = sanitize_input($_POST['username'] ?? '');
    $email = sanitize_input($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $roleIds = $_POST['roles'] ?? [];

    if (empty($username) || empty($email) || empty($password)) {
        set_flash_message('Заповніть всі обов\'язкові поля', 'danger');
        return;
    }

    if (strlen($password) < 8) {
        set_flash_message('Пароль має бути не менше 8 символів', 'danger');
        return;
    }

    // Проверяем существование username
    $existing = db_fetch_one('SELECT id FROM users WHERE username = ?', [$username]);
    if ($existing) {
        set_flash_message('Користувач з таким логіном вже існує', 'danger');
        return;
    }

    // Проверяем существование email
    $existing = db_fetch_one('SELECT id FROM users WHERE email = ?', [$email]);
    if ($existing) {
        set_flash_message('Користувач з таким email вже існує', 'danger');
        return;
    }

    // Хешируем пароль
    $passwordHash = password_hash($password, PASSWORD_DEFAULT);

    // Подготовка role_ids
    $roleIdsJson = !empty($roleIds) && is_array($roleIds)
        ? json_encode(array_map('intval', $roleIds))
        : null;

    // Создаем пользователя
    $stmt = db_query('
        INSERT INTO users (username, email, password, role_ids, is_active)
        VALUES (?, ?, ?, ?, 1)
    ', [$username, $email, $passwordHash, $roleIdsJson]);

    if ($stmt) {
        set_flash_message('Користувача успішно створено', 'success');
        admin_redirect('users');
    } else {
        set_flash_message('Помилка при створенні користувача', 'danger');
    }
}

/**
 * Обработка обновления пользователя
 */
function handle_update_user() {
    $session = function_exists('sessionManager') ? sessionManager() : null;
    $userId = $session ? (int)($session->get('admin_user_id') ?? 0) : 0;

    $hasAccess = ($userId === 0 || $userId === 1) || (function_exists('current_user_can') && current_user_can('admin.access'));
    if (!$hasAccess) {
        set_flash_message('У вас немає прав на редагування користувачів', 'danger');
        return;
    }

    $userIdToUpdate = (int)($_POST['user_id'] ?? 0);
    $email = sanitize_input($_POST['email'] ?? '');
    $isActive = isset($_POST['is_active']) ? 1 : 0;
    $roleIds = $_POST['roles'] ?? [];

    if (empty($userIdToUpdate) || empty($email)) {
        set_flash_message('Невірні дані', 'danger');
        return;
    }

    // Нельзя деактивировать первого пользователя
    if ($userIdToUpdate === 1 && $isActive === 0) {
        set_flash_message('Неможливо деактивувати першого користувача', 'danger');
        return;
    }

    // Проверяем email
    $existing = db_fetch_one('SELECT id FROM users WHERE email = ? AND id != ?', [$email, $userIdToUpdate]);
    if ($existing) {
        set_flash_message('Email вже використовується іншим користувачем', 'danger');
        return;
    }

    // Подготовка role_ids
    $roleIdsJson = !empty($roleIds) && is_array($roleIds)
        ? json_encode(array_map('intval', $roleIds))
        : null;

    // Обновляем пользователя
    $stmt = db_query('
        UPDATE users
        SET email = ?, is_active = ?, role_ids = ?
        WHERE id = ?
    ', [$email, $isActive, $roleIdsJson, $userIdToUpdate]);

    if ($stmt) {
        set_flash_message('Користувача успішно оновлено', 'success');
        admin_redirect('users');
    } else {
        set_flash_message('Помилка при оновленні користувача', 'danger');
    }
}

/**
 * Обработка удаления пользователя
 */
function handle_delete_user() {
    $session = function_exists('sessionManager') ? sessionManager() : null;
    $userId = $session ? (int)($session->get('admin_user_id') ?? 0) : 0;

    $hasAccess = ($userId === 0 || $userId === 1) || (function_exists('current_user_can') && current_user_can('admin.access'));
    if (!$hasAccess) {
        set_flash_message('У вас немає прав на видалення користувачів', 'danger');
        return;
    }

    $userIdToDelete = (int)($_POST['user_id'] ?? 0);

    if (empty($userIdToDelete)) {
        set_flash_message('Невірні дані', 'danger');
        return;
    }

    // Нельзя удалить первого пользователя
    if ($userIdToDelete === 1) {
        set_flash_message('Неможливо видалити першого користувача', 'danger');
        return;
    }

    // Нельзя удалить себя
    if ($userIdToDelete === $userId) {
        set_flash_message('Ви не можете видалити себе', 'danger');
        return;
    }

    $stmt = db_query('DELETE FROM users WHERE id = ?', [$userIdToDelete]);

    if ($stmt) {
        set_flash_message('Користувача успішно видалено', 'success');
        admin_redirect('users');
    } else {
        set_flash_message('Помилка при видаленні користувача', 'danger');
    }
}

/**
 * Обработка смены пароля
 */
function handle_change_password() {
    $session = function_exists('sessionManager') ? sessionManager() : null;
    $userId = $session ? (int)($session->get('admin_user_id') ?? 0) : 0;

    $hasAccess = ($userId === 0 || $userId === 1) || (function_exists('current_user_can') && current_user_can('admin.access'));
    if (!$hasAccess) {
        set_flash_message('У вас немає прав на зміну паролів', 'danger');
        return;
    }

    $userIdToUpdate = (int)($_POST['user_id'] ?? 0);
    $password = $_POST['password'] ?? '';

    if (empty($userIdToUpdate) || empty($password)) {
        set_flash_message('Невірні дані', 'danger');
        return;
    }

    if (strlen($password) < 8) {
        set_flash_message('Пароль має бути не менше 8 символів', 'danger');
        return;
    }

    $passwordHash = password_hash($password, PASSWORD_DEFAULT);
    $stmt = db_query('UPDATE users SET password = ? WHERE id = ?', [$passwordHash, $userIdToUpdate]);

    if ($stmt) {
        set_flash_message('Пароль успішно змінено', 'success');
        admin_redirect('users');
    } else {
        set_flash_message('Помилка при зміні пароля', 'danger');
    }
}

/**
 * Вспомогательные функции
 */
// Функции sanitize_input и set_flash_message определены в content/admin/includes/functions.php
