<?php
/**
 * Страница управления темами (функция вместо класса)
 *
 * Примечание: Авторизация обрабатывается автоматически через middleware 'admin.auth'
 * Разработчикам не нужно проверять авторизацию вручную - фреймворк делает это автоматически
 */

function admin_themes_page() {
    // Обработка POST запросов (действия с темами)
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST['action'])) {
        handle_theme_action();
        // Редирект после обработки действия
        if (function_exists('admin_redirect')) {
            admin_redirect('themes');
        } else {
            header('Location: ' . admin_url('themes'));
        }
        exit;
    }

    // Обработка AJAX запросов
    if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) &&
        strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') {
        handle_themes_ajax();
        return;
    }

    // Получаем данные тем через хуки
    $data = get_content_data('themes_data', [
        'themes' => get_themes_list(),
    ]);

    // Создаем кнопку "Загрузить тему"
    $uploadButton = '';
    if (file_exists(__DIR__ . DS . '..' . DS . 'components' . DS . 'button' . DS . 'button.php')) {
        ob_start();
        $text = 'Завантажити тему';
        $type = 'primary';
        $icon = 'upload';
        $attributes = [
            'data-bs-toggle' => 'modal',
            'data-bs-target' => '#uploadThemeModal',
            'onclick' => 'window.ModalHandler && window.ModalHandler.show("uploadThemeModal")',
        ];
        include __DIR__ . DS . '..' . DS . 'components' . DS . 'button' . DS . 'button.php';
        $uploadButton = ob_get_clean();
    } else {
        // Fallback кнопка
        $uploadButton = '<button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#uploadThemeModal">
            <i class="fas fa-upload"></i> Завантажити тему
        </button>';
    }

    $layout = [
        'title' => 'Керування темами - Flowaxy CMS',
        'content' => render_themes_content($data),
        'pageHeaderIcon' => 'fas fa-palette',
        'pageHeaderButtons' => $uploadButton,
        'pageBreadcrumbs' => [
            ['title' => 'Головна', 'url' => admin_url('dashboard'), 'page' => 'dashboard'],
            ['title' => 'Керування темами', 'page' => 'themes'],
        ],
        'additionalJS' => [
            asset('scripts/themes.js'),
        ],
    ];

    return render_admin_layout($layout);
}

function render_themes_content($data) {
    ob_start();
    $templatePath = __DIR__ . DS . 'themes' . DS . 'template.php';
    if (file_exists($templatePath)) {
        // Определяем путь к компонентам
        $componentsPath = __DIR__ . DS . '..' . DS . 'components' . DS;

        // Получаем список тем и статистику
        $themes = $data['themes'] ?? [];
        $installedThemes = [];
        $stats = [
            'total' => 0,
            'installed' => 0,
            'active' => 0,
            'inactive' => 0,
        ];

        if (is_array($themes) && !empty($themes)) {
            $stats['total'] = count($themes);
            foreach ($themes as $theme) {
                if (is_array($theme)) {
                    // Убеждаемся, что slug есть в массиве темы
                    if (empty($theme['slug'])) {
                        continue;
                    }

                    // Получаем реальный статус темы из themes.ini
                    $themeStatus = 'non-install';
                    if (class_exists(\Flowaxy\Infrastructure\Config\ThemeConfigManager::class)) {
                        $themeStatus = \Flowaxy\Infrastructure\Config\ThemeConfigManager::getStatus($theme['slug']);
                    }

                    // Определяем статусы на основе themes.ini
                    $isActive = ($themeStatus === 'activated');
                    $isInstalled = ($themeStatus !== 'non-install');
                    $isDeactivated = ($themeStatus === 'deactivated');

                    // Добавляем статусы для компонента theme-card
                    $theme['is_active'] = $isActive ? 1 : 0;
                    $theme['is_installed'] = $isInstalled;
                    $theme['status'] = $themeStatus; // Добавляем статус для использования в компоненте

                    // Получаем зависимости темы из метаданных
                    $theme['requires_plugins'] = [];
                    if (class_exists(\Flowaxy\Support\Helpers\ThemeMetadataHelper::class)) {
                        $metadata = \Flowaxy\Support\Helpers\ThemeMetadataHelper::readMetadata($theme['slug'] ?? '');
                        if (isset($metadata['requires_plugins']) && is_array($metadata['requires_plugins']) && !empty($metadata['requires_plugins'])) {
                            $theme['requires_plugins'] = $metadata['requires_plugins'];
                        }
                    }

                    // Добавляем все темы в список (включая non-install)
                    $installedThemes[] = $theme;

                    // Подсчитываем статистику
                    if ($isInstalled) {
                        $stats['installed']++;
                        if ($isActive) {
                            $stats['active']++;
                        } else {
                            $stats['inactive']++;
                        }
                    }
                }
            }
        }

        $data['componentsPath'] = $componentsPath;
        $data['installedThemes'] = $installedThemes;
        $data['stats'] = $stats;

        extract($data);
        include $templatePath;
    }
    return ob_get_clean();
}

function get_themes_list() {
    $themes = [];
    if (function_exists('themeManager')) {
        $themeManager = themeManager();
        if ($themeManager && method_exists($themeManager, 'getAllThemes')) {
            $themes = $themeManager->getAllThemes();
        }
    }
    return $themes;
}

function handle_theme_action() {
    // Проверка CSRF токена
    if (!function_exists('verify_csrf_token') || !verify_csrf_token()) {
        if (function_exists('set_flash_message')) {
            set_flash_message('Помилка безпеки', 'danger');
        }
        return;
    }

    // Проверка прав доступа
    if (!function_exists('current_user_can') || !current_user_can('admin.access')) {
        if (function_exists('set_flash_message')) {
            set_flash_message('У вас немає прав на виконання цієї дії', 'danger');
        }
        return;
    }

    $action = $_POST['action'] ?? '';
    $themeSlug = $_POST['theme_slug'] ?? '';

    // Для действий clear_themes_cache и upload_theme не требуется theme_slug
    $actionsWithoutSlug = ['clear_themes_cache', 'upload_theme'];
    if (!in_array($action, $actionsWithoutSlug) && (empty($action) || empty($themeSlug))) {
        if (function_exists('set_flash_message')) {
            set_flash_message('Не вказано дію або slug теми', 'danger');
        }
        return;
    }

    try {
        if (!function_exists('themeManager')) {
            if (function_exists('set_flash_message')) {
                set_flash_message('ThemeManager недоступний', 'danger');
            }
            return;
        }

        $themeManager = themeManager();
        if (!$themeManager) {
            if (function_exists('set_flash_message')) {
                set_flash_message('ThemeManager недоступний', 'danger');
            }
            return;
        }

        switch ($action) {
            case 'activate_theme':
                if (method_exists($themeManager, 'activateTheme')) {
                    $result = $themeManager->activateTheme($themeSlug);
                    if ($result) {
                        if (function_exists('set_flash_message')) {
                            set_flash_message('Тему активовано', 'success');
                        }
                    } else {
                        // Отримуємо детальну помилку, якщо доступна
                        $errorMessage = 'Помилка активації теми';
                        if (method_exists($themeManager, 'getLastActivationError')) {
                            $lastError = $themeManager->getLastActivationError();
                            if (!empty($lastError)) {
                                $errorMessage = $lastError;
                            }
                        }
                        if (function_exists('set_flash_message')) {
                            set_flash_message($errorMessage, 'danger');
                        }
                    }
                }
                break;

            case 'deactivate_theme':
                if (method_exists($themeManager, 'deactivateTheme')) {
                    $result = $themeManager->deactivateTheme($themeSlug);
                    if ($result) {
                        if (function_exists('set_flash_message')) {
                            set_flash_message('Тему деактивовано', 'success');
                        }
                    } else {
                        if (function_exists('set_flash_message')) {
                            set_flash_message('Помилка деактивації теми', 'danger');
                        }
                    }
                }
                break;

            case 'install_theme':
                // Получаем информацию о теме для установки
                $theme = $themeManager->getTheme($themeSlug);
                if ($theme && is_array($theme)) {
                    $name = $theme['name'] ?? $themeSlug;
                    $description = $theme['description'] ?? '';
                    $version = $theme['version'] ?? '1.0.0';
                    $author = $theme['author'] ?? '';

                    if (method_exists($themeManager, 'installTheme')) {
                        $result = $themeManager->installTheme($themeSlug, $name, $description, $version, $author);
                        if ($result) {
                            if (function_exists('set_flash_message')) {
                                set_flash_message('Тему встановлено', 'success');
                            }
                        } else {
                            if (function_exists('set_flash_message')) {
                                set_flash_message('Помилка встановлення теми', 'danger');
                            }
                        }
                    }
                } else {
                    if (function_exists('set_flash_message')) {
                        set_flash_message('Тему не знайдено', 'danger');
                    }
                }
                break;

            case 'uninstall_theme':
                if (method_exists($themeManager, 'uninstallTheme')) {
                    $result = $themeManager->uninstallTheme($themeSlug);
                    if ($result) {
                        if (function_exists('set_flash_message')) {
                            set_flash_message('Тему видалено', 'success');
                        }
                    } else {
                        if (function_exists('set_flash_message')) {
                            set_flash_message('Помилка видалення теми', 'danger');
                        }
                    }
                } else {
                    if (function_exists('set_flash_message')) {
                        set_flash_message('Метод видалення теми не знайдено', 'danger');
                    }
                }
                break;

            case 'clear_themes_cache':
                if (method_exists($themeManager, 'clearThemeCache')) {
                    $themeManager->clearThemeCache();
                    if (function_exists('set_flash_message')) {
                        set_flash_message('Кеш тем очищено', 'success');
                    }
                } else {
                    if (function_exists('set_flash_message')) {
                        set_flash_message('Метод очистки кешу не знайдено', 'danger');
                    }
                }
                break;

            case 'upload_theme':
                // Обработка загрузки темы из ZIP-архива
                handle_theme_upload();
                break;

            default:
                if (function_exists('set_flash_message')) {
                    set_flash_message('Невідома дія', 'danger');
                }
        }
    } catch (\Throwable $e) {
        if (function_exists('error_log')) {
            error_log('handle_theme_action error: ' . $e->getMessage() . ' | Trace: ' . $e->getTraceAsString());
        }
        if (function_exists('set_flash_message')) {
            set_flash_message('Помилка: ' . $e->getMessage(), 'danger');
        }
    }
}

function handle_theme_upload() {
    // Проверка наличия загруженных файлов
    if (empty($_FILES['theme_file']) || !is_array($_FILES['theme_file']['name'])) {
        if (function_exists('set_flash_message')) {
            set_flash_message('Файли не вибрано', 'danger');
        }
        return;
    }

    // Загружаем необходимые классы
    if (!class_exists(\Flowaxy\Support\Services\FileUploadService::class, false)) {
        \Flowaxy\Support\Helpers\ClassLoaderHelper::ensureLoaded(\Flowaxy\Support\Services\FileUploadService::class);
    }
    if (!class_exists(\Flowaxy\Support\Services\ArchiveExtractionService::class, false)) {
        \Flowaxy\Support\Helpers\ClassLoaderHelper::ensureLoaded(\Flowaxy\Support\Services\ArchiveExtractionService::class);
    }
    if (!class_exists(\Flowaxy\Core\System\PathResolver::class, false)) {
        \Flowaxy\Support\Helpers\ClassLoaderHelper::ensureLoaded(\Flowaxy\Core\System\PathResolver::class);
    }

    $uploadService = new \Flowaxy\Support\Services\FileUploadService();
    $archiveService = new \Flowaxy\Support\Services\ArchiveExtractionService();
    $themesDir = \Flowaxy\Core\System\PathResolver::themes();

    $uploadedCount = 0;
    $errors = [];
    $uploadedThemes = []; // Список успешно загруженных тем

    // Обрабатываем каждый загруженный файл
    $files = $_FILES['theme_file'];
    $fileCount = count($files['name']);

    for ($i = 0; $i < $fileCount; $i++) {
        // Формируем массив файла для обработки
        $file = [
            'name' => $files['name'][$i],
            'type' => $files['type'][$i],
            'tmp_name' => $files['tmp_name'][$i],
            'error' => $files['error'][$i],
            'size' => $files['size'][$i],
        ];

        // Пропускаем файлы с ошибками
        if ($file['error'] !== UPLOAD_ERR_OK) {
            $errors[] = "Файл '{$file['name']}': помилка завантаження (код: {$file['error']})";
            continue;
        }

        try {
            // Загружаем файл во временную директорию
            $uploadResult = $uploadService->upload($file, [
                'allowed_extensions' => ['zip'],
                'allowed_mime_types' => ['application/zip', 'application/x-zip-compressed'],
                'max_file_size' => 50 * 1024 * 1024, // 50 MB
            ]);

            if (!$uploadResult['success']) {
                $errors[] = "Файл '{$file['name']}': " . ($uploadResult['error'] ?? 'Помилка завантаження');
                continue;
            }

            $uploadedFile = $uploadResult['file'] ?? null;
            if (!$uploadedFile || !file_exists($uploadedFile)) {
                $errors[] = "Файл '{$file['name']}': файл не знайдено після завантаження";
                continue;
            }

            // Сначала валидируем архив и получаем slug
            // Пробуем с Theme.php (с большой буквы)
            $zip = \Flowaxy\Support\Facades\Archive::open($uploadedFile, \ZipArchive::RDONLY);
            $validation = $archiveService->validateArchive($zip, 'Theme.php');
            $zip->close();

            // Если не нашли Theme.php, пробуем theme.php (с маленькой буквы)
            if (!$validation['valid'] && str_contains($validation['error'] ?? '', 'не містить')) {
                $zip = \Flowaxy\Support\Facades\Archive::open($uploadedFile, \ZipArchive::RDONLY);
                if ($zip) {
                    $validation = $archiveService->validateArchive($zip, 'theme.php');
                    $zip->close();
                }
            }

            if (!$validation['valid']) {
                $errors[] = "Файл '{$file['name']}': " . ($validation['error'] ?? 'Архів не валідний');
                // Очищаем загруженный файл
                if ($uploadedFile && file_exists($uploadedFile)) {
                    \Flowaxy\Support\Helpers\ResourceCleanupHelper::cleanupTempFiles($uploadedFile);
                }
                continue;
            }

            $slug = $validation['slug'] ?? null;
            if (empty($slug)) {
                $errors[] = "Файл '{$file['name']}': неможливо визначити slug теми";
                // Очищаем загруженный файл
                if ($uploadedFile && file_exists($uploadedFile)) {
                    \Flowaxy\Support\Helpers\ResourceCleanupHelper::cleanupTempFiles($uploadedFile);
                }
                continue;
            }

            // Определяем путь назначения для темы
            $ds = defined('DS') ? DS : DIRECTORY_SEPARATOR;
            $themeDestinationPath = $themesDir . $ds . $slug;

            // Извлекаем архив в директорию темы
            // Пробуем с Theme.php (с большой буквы)
            $extractionResult = $archiveService->extractZip(
                $uploadedFile,
                $themeDestinationPath,
                'Theme.php',
                false // не перезаписывать существующие темы
            );

            // Если не нашли Theme.php, пробуем theme.php (с маленькой буквы)
            if (!$extractionResult['success'] && str_contains($extractionResult['error'] ?? '', 'не містить')) {
                $extractionResult = $archiveService->extractZip(
                    $uploadedFile,
                    $themeDestinationPath,
                    'theme.php',
                    false
                );
            }

            // Очищаем загруженный файл
            if ($uploadedFile && file_exists($uploadedFile)) {
                \Flowaxy\Support\Helpers\ResourceCleanupHelper::cleanupTempFiles($uploadedFile);
            }

            if (!$extractionResult['success']) {
                $errors[] = "Файл '{$file['name']}': " . ($extractionResult['error'] ?? 'Помилка витягування архіву');
                continue;
            }

            $slug = $extractionResult['slug'] ?? null;
            if (empty($slug)) {
                $errors[] = "Файл '{$file['name']}': неможливо визначити slug теми";
                continue;
            }

            // Тема успешно загружена и извлечена, но не устанавливается автоматически
            // Пользователь должен установить её вручную через интерфейс
            $uploadedCount++;
            $uploadedThemes[] = $slug; // Сохраняем slug загруженной темы
            try {
                if (class_exists('\Flowaxy\Support\Facades\Log')) {
                    \Flowaxy\Support\Facades\Log::Info('Тему завантажено', [
                        'slug' => $slug,
                        'file' => $file['name'],
                    ]);
                }
            } catch (\Throwable $logError) {
                // Ignore logging errors
            }
        } catch (\Throwable $e) {
            $errors[] = "Файл '{$file['name']}': " . $e->getMessage();
            try {
                if (class_exists('\Flowaxy\Support\Facades\Log')) {
                    \Flowaxy\Support\Facades\Log::Error('Помилка завантаження теми', [
                        'exception' => $e,
                        'file' => $file['name'],
                    ]);
                }
            } catch (\Throwable $logError) {
                // Ignore logging errors
            }
        }
    }

    // Формируем сообщение о результате
    if ($uploadedCount > 0) {
        // Очищаем кеш тем после успешной загрузки
        if (class_exists('\Flowaxy\Support\Helpers\CacheHelper')) {
            try {
                \Flowaxy\Support\Helpers\CacheHelper::forget('all_themes_filesystem');
                // Очищаем кеш для каждой загруженной темы
                foreach ($uploadedThemes as $slug) {
                    \Flowaxy\Support\Helpers\CacheHelper::forget('theme_' . $slug);
                    \Flowaxy\Support\Helpers\CacheHelper::forget('theme_config_' . $slug);
                }
            } catch (\Throwable $cacheError) {
                // Игнорируем ошибки очистки кеша
            }
        }

        // Также очищаем кеш через ThemeManager, если доступен
        if (function_exists('themeManager')) {
            $themeManager = themeManager();
            if ($themeManager && method_exists($themeManager, 'clearThemeCache')) {
                try {
                    $themeManager->clearThemeCache();
                } catch (\Throwable $cacheError) {
                    // Игнорируем ошибки очистки кеша
                }
            }
        }

        $themeNames = implode(', ', $uploadedThemes);
        $message = "Успішно завантажено тем: {$uploadedCount}";
        if (count($uploadedThemes) > 0) {
            $message .= " (" . $themeNames . ")";
        }
        if (function_exists('set_flash_message')) {
            set_flash_message($message, 'success');
        }
    }

    if (!empty($errors)) {
        $errorMessage = 'Помилки завантаження: ' . implode('; ', $errors);
        if (function_exists('set_flash_message')) {
            set_flash_message($errorMessage, 'danger');
        }
    }

    // Если ничего не загружено и нет ошибок, значит файлы не были выбраны
    if ($uploadedCount === 0 && empty($errors)) {
        if (function_exists('set_flash_message')) {
            set_flash_message('Файли не вибрано', 'warning');
        }
    }
}

function handle_themes_ajax() {
    // Обработка AJAX запросов для тем
    // Реализация будет добавлена позже
    if (function_exists('send_json_response')) {
        send_json_response(['success' => false, 'error' => 'Not implemented'], 501);
    } else {
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'error' => 'Not implemented']);
        exit;
    }
}
