<?php
/**
 * Страница настроек сайта (функция вместо класса)
 */

use Flowaxy\Infrastructure\Config\LoggingConfigManager;
use Flowaxy\Support\Managers\SettingsManager;

function admin_site_settings_page() {
    // Проверка авторизации
    if (!function_exists('current_user_can') || !current_user_can('admin.settings.manage')) {
        if (function_exists('admin_redirect')) {
            admin_redirect('login');
        }
        return;
    }

    // Обработка POST запросов
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_settings'])) {
        handle_save_site_settings();
    }

    // Получаем данные через хуки
    $data = get_content_data('site_settings_data', [
        'settings' => get_site_settings(),
    ]);

    $layout = [
        'title' => 'Налаштування сайту - Flowaxy CMS',
        'content' => render_site_settings_content($data),
        'pageHeaderIcon' => 'fas fa-globe',
        'pageBreadcrumbs' => [
            ['title' => 'Головна', 'url' => admin_url('dashboard'), 'page' => 'dashboard'],
            ['title' => 'Налаштування', 'url' => admin_url('settings'), 'page' => 'settings'],
            ['title' => 'Налаштування сайту'],
        ],
        'additionalJS' => [
            asset('scripts/site-settings.js'),
        ],
    ];

    return render_admin_layout($layout);
}

function render_site_settings_content($data) {
    ob_start();
    $templatePath = __DIR__ . DS . 'site-settings' . DS . 'template.php';
    if (file_exists($templatePath)) {
        extract($data);
        include $templatePath;
    }
    return ob_get_clean();
}

function get_site_settings() {
    // Получаем настройки сайта из SettingsManager
    $settings = [];

    if (class_exists(SettingsManager::class)) {
        $settingsManager = SettingsManager::getInstance();
        $settings = $settingsManager->all();
    }

    // Получаем настройки логирования из LoggingConfigManager
    if (class_exists(LoggingConfigManager::class)) {
        $loggingConfig = LoggingConfigManager::loadConfig();

        // Преобразуем ключи для совместимости с формой
        $settings['logging_enabled'] = isset($loggingConfig['enabled']) ? (string)$loggingConfig['enabled'] : '1';

        // Преобразуем строку levels в массив для совместимости с шаблоном
        $levelsString = isset($loggingConfig['levels']) ? (string)$loggingConfig['levels'] : 'DEBUG,INFO,WARNING,ERROR,CRITICAL';
        // Если строка пустая (все отключено), возвращаем пустой массив
        $settings['logging_levels'] = !empty($levelsString) ? explode(',', $levelsString) : [];

        // Преобразуем строку types в массив для совместимости с шаблоном
        $typesString = isset($loggingConfig['types']) ? (string)$loggingConfig['types'] : 'file';
        // Если строка пустая (все отключено), возвращаем пустой массив
        $settings['logging_types'] = !empty($typesString) ? explode(',', $typesString) : [];

        $settings['logging_max_file_size'] = isset($loggingConfig['max_file_size']) ? (string)$loggingConfig['max_file_size'] : '10485760';
        $settings['logging_retention_days'] = isset($loggingConfig['retention_days']) ? (string)$loggingConfig['retention_days'] : '30';
        $settings['logging_rotation_type'] = isset($loggingConfig['rotation_type']) ? (string)$loggingConfig['rotation_type'] : 'size';
        $settings['logging_rotation_time'] = isset($loggingConfig['rotation_time']) ? (string)$loggingConfig['rotation_time'] : '24';
        $settings['logging_rotation_time_unit'] = isset($loggingConfig['rotation_time_unit']) ? (string)$loggingConfig['rotation_time_unit'] : 'hours';
    }

    // Применяем фильтры для совместимости
    if (function_exists('apply_filters')) {
        $settings = apply_filters('site_settings', $settings);
    }

    return $settings;
}

function handle_save_site_settings() {
    // Логируем начало обработки
    try {
        if (class_exists('\Flowaxy\Support\Facades\Log')) {
            \Flowaxy\Support\Facades\Log::Debug('handle_save_site_settings: Начало обработки сохранения настроек', [
                'post_data' => array_keys($_POST),
                'has_settings' => isset($_POST['settings']),
            ]);
        }
    } catch (\Throwable $e) {
        // Ignore logging errors
    }

    // Проверка CSRF токена
    $csrfToken = $_POST['csrf_token'] ?? null;
    if (!function_exists('verify_csrf_token') || !verify_csrf_token($csrfToken)) {
        try {
            if (class_exists('\Flowaxy\Support\Facades\Log')) {
                \Flowaxy\Support\Facades\Log::Warning('handle_save_site_settings: CSRF токен не прошел проверку', [
                    'token_provided' => !empty($csrfToken),
                    'function_exists' => function_exists('verify_csrf_token'),
                ]);
            }
        } catch (\Throwable $e) {
            // Ignore logging errors
        }
        set_flash_message('Помилка безпеки. Спробуйте ще раз.', 'error');
        admin_redirect('site-settings');
        return;
    }

    if (!isset($_POST['settings']) || !is_array($_POST['settings'])) {
        try {
            if (class_exists('\Flowaxy\Support\Facades\Log')) {
                \Flowaxy\Support\Facades\Log::Warning('handle_save_site_settings: Данные не получены', [
                    'has_post' => isset($_POST),
                    'has_settings' => isset($_POST['settings']),
                ]);
            }
        } catch (\Throwable $e) {
            // Ignore logging errors
        }
        set_flash_message('Помилка: дані не отримано', 'error');
        admin_redirect('site-settings');
        return;
    }

    $postSettings = $_POST['settings'];

    // Разделяем настройки: логирование отдельно, остальное в site.php
    $loggingSettings = [];
    $siteSettings = [];

    // Ключи настроек логирования
    $loggingKeys = [
        'logging_enabled',
        'logging_levels',
        'logging_types',
        'logging_max_file_size',
        'logging_retention_days',
        'logging_rotation_type',
        'logging_rotation_time',
        'logging_rotation_time_unit',
    ];

    foreach ($postSettings as $key => $value) {
        if (in_array($key, $loggingKeys, true)) {
            // Настройки логирования сохраняем в logging.ini
            $loggingSettings[$key] = $value;
        } else {
            // Остальные настройки сохраняем в site.php
            $siteSettings[$key] = $value;
        }
    }

    // Сохраняем настройки логирования в logging.ini
    // Всегда сохраняем настройки логирования, даже если они не были изменены
    if (class_exists(LoggingConfigManager::class)) {
        // Загружаем текущие настройки для сохранения неизмененных значений
        $currentConfig = LoggingConfigManager::loadConfig();
        $loggingConfig = [];

        // Преобразуем ключи для logging.ini
        if (isset($loggingSettings['logging_enabled'])) {
            $loggingConfig['enabled'] = $loggingSettings['logging_enabled'] === '1' ? '1' : '0';
        } elseif (isset($currentConfig['enabled'])) {
            $loggingConfig['enabled'] = (string)$currentConfig['enabled'];
        }

        // Обрабатываем уровни логирования
        // ВАЖНО: Если ключ присутствует в POST (даже если это пустой массив), используем его значение
        if (array_key_exists('logging_levels', $loggingSettings)) {
            if (is_array($loggingSettings['logging_levels'])) {
                // Массив может быть пустым (все чекбоксы отключены) или содержать выбранные значения
                $loggingConfig['levels'] = !empty($loggingSettings['logging_levels'])
                    ? implode(',', $loggingSettings['logging_levels'])
                    : '';
            } else {
                // Если это не массив, преобразуем в строку
                $loggingConfig['levels'] = (string)$loggingSettings['logging_levels'];
            }
        } elseif (isset($currentConfig['levels'])) {
            // Если ключ не был передан, используем текущее значение
            $loggingConfig['levels'] = (string)$currentConfig['levels'];
        }

        // Обрабатываем типы логирования
        // ВАЖНО: Если ключ присутствует в POST (даже если это пустой массив), используем его значение
        if (array_key_exists('logging_types', $loggingSettings)) {
            if (is_array($loggingSettings['logging_types'])) {
                // Массив может быть пустым (все чекбоксы отключены) или содержать выбранные значения
                $loggingConfig['types'] = !empty($loggingSettings['logging_types'])
                    ? implode(',', $loggingSettings['logging_types'])
                    : '';
            } else {
                // Если это не массив, преобразуем в строку
                $loggingConfig['types'] = (string)$loggingSettings['logging_types'];
            }
        } elseif (isset($currentConfig['types'])) {
            // Если ключ не был передан, используем текущее значение
            $loggingConfig['types'] = (string)$currentConfig['types'];
        }

        if (isset($loggingSettings['logging_max_file_size'])) {
            $loggingConfig['max_file_size'] = (string)$loggingSettings['logging_max_file_size'];
        } elseif (isset($currentConfig['max_file_size'])) {
            $loggingConfig['max_file_size'] = (string)$currentConfig['max_file_size'];
        }

        if (isset($loggingSettings['logging_retention_days'])) {
            $loggingConfig['retention_days'] = (string)$loggingSettings['logging_retention_days'];
        } elseif (isset($currentConfig['retention_days'])) {
            $loggingConfig['retention_days'] = (string)$currentConfig['retention_days'];
        }

        if (isset($loggingSettings['logging_rotation_type'])) {
            $loggingConfig['rotation_type'] = (string)$loggingSettings['logging_rotation_type'];
        } elseif (isset($currentConfig['rotation_type'])) {
            $loggingConfig['rotation_type'] = (string)$currentConfig['rotation_type'];
        }

        if (isset($loggingSettings['logging_rotation_time'])) {
            $loggingConfig['rotation_time'] = (string)$loggingSettings['logging_rotation_time'];
        } elseif (isset($currentConfig['rotation_time'])) {
            $loggingConfig['rotation_time'] = (string)$currentConfig['rotation_time'];
        }

        if (isset($loggingSettings['logging_rotation_time_unit'])) {
            $loggingConfig['rotation_time_unit'] = (string)$loggingSettings['logging_rotation_time_unit'];
        } elseif (isset($currentConfig['rotation_time_unit'])) {
            $loggingConfig['rotation_time_unit'] = (string)$currentConfig['rotation_time_unit'];
        }

        // Всегда сохраняем настройки, даже если они не были изменены
        // Убеждаемся, что все обязательные поля присутствуют
        if (empty($loggingConfig)) {
            // Если ничего не было передано, загружаем текущие настройки
            $loggingConfig = $currentConfig;
        }

        // Логируем перед сохранением
        try {
            if (class_exists('\Flowaxy\Support\Facades\Log')) {
                \Flowaxy\Support\Facades\Log::Debug('handle_save_site_settings: Сохранение настроек логирования', [
                    'logging_config' => $loggingConfig,
                    'logging_settings_keys' => array_keys($loggingSettings),
                ]);
            }
        } catch (\Throwable $e) {
            // Ignore logging errors
        }

        $loggingSaved = LoggingConfigManager::saveConfig($loggingConfig);
        if (!$loggingSaved) {
            try {
                if (class_exists('\Flowaxy\Support\Facades\Log')) {
                    \Flowaxy\Support\Facades\Log::Error('handle_save_site_settings: Ошибка сохранения настроек логирования', [
                        'logging_config' => $loggingConfig,
                    ]);
                }
            } catch (\Throwable $e) {
                // Ignore logging errors
            }
            set_flash_message('Помилка збереження налаштувань логування', 'error');
            admin_redirect('site-settings');
            return;
        }

        // Логируем успешное сохранение
        try {
            if (class_exists('\Flowaxy\Support\Facades\Log')) {
                \Flowaxy\Support\Facades\Log::Info('handle_save_site_settings: Настройки логирования успешно сохранены', [
                    'logging_config' => $loggingConfig,
                ]);
            }
        } catch (\Throwable $e) {
            // Ignore logging errors
        }
    }

    // Сохраняем остальные настройки в site.php
    if (!empty($siteSettings) && class_exists(SettingsManager::class)) {
        $settingsManager = SettingsManager::getInstance();
        $siteSaved = $settingsManager->setMultiple($siteSettings);
        if (!$siteSaved) {
            try {
                if (class_exists('\Flowaxy\Support\Facades\Log')) {
                    \Flowaxy\Support\Facades\Log::Error('handle_save_site_settings: Ошибка сохранения настроек сайта', [
                        'site_settings_keys' => array_keys($siteSettings),
                    ]);
                }
            } catch (\Throwable $e) {
                // Ignore logging errors
            }
            set_flash_message('Помилка збереження налаштувань сайту', 'error');
            admin_redirect('site-settings');
            return;
        }
    }

    // Логируем успешное завершение
    try {
        if (class_exists('\Flowaxy\Support\Facades\Log')) {
            \Flowaxy\Support\Facades\Log::Info('handle_save_site_settings: Все настройки успешно сохранены', [
                'logging_settings_count' => count($loggingSettings),
                'site_settings_count' => count($siteSettings),
            ]);
        }
    } catch (\Throwable $e) {
        // Ignore logging errors
    }

    set_flash_message('Налаштування збережено', 'success');
    admin_redirect('site-settings');
}
