<?php
/**
 * Страница управления ролями и правами (функция вместо класса)
 */

/**
 * Страница управления ролями и правами
 *
 * Примечание: Авторизация обрабатывается автоматически через middleware 'admin.auth'
 * Разработчикам не нужно проверять авторизацию вручную - фреймворк делает это автоматически
 */
function admin_roles_page() {
    // Проверка подключения к БД
    if (!function_exists('is_database_available') || !is_database_available()) {
        $layout = [
            'title' => 'Ролі та права - Flowaxy CMS',
            'content' => render_database_required_message('Ролі та права'),
            'pageHeaderIcon' => 'fas fa-user-shield',
            'pageBreadcrumbs' => [
                ['title' => 'Головна', 'url' => admin_url('dashboard')],
                ['title' => 'Налаштування', 'url' => admin_url('settings')],
                ['title' => 'Ролі та права'],
            ],
        ];
        return render_admin_layout($layout);
    }

    // Обработка POST запросов
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create_role':
                handle_create_role();
                break;
            case 'update_role':
                handle_update_role();
                break;
            case 'delete_role':
                handle_delete_role();
                break;
        }
    }

    // Получаем данные через хуки
    $data = get_content_data('roles_data', [
        'roles' => get_roles(),
        'permissions' => get_all_permissions(),
    ]);

    // Рендерим страницу
    $layout = [
        'title' => 'Ролі та права - Flowaxy CMS',
        'content' => render_roles_content($data),
        'pageHeaderIcon' => 'fas fa-user-shield',
        'pageBreadcrumbs' => [
            ['title' => 'Головна', 'url' => admin_url('dashboard'), 'page' => 'dashboard'],
            ['title' => 'Налаштування', 'url' => admin_url('settings'), 'page' => 'settings'],
            ['title' => 'Ролі та права', 'page' => 'roles'],
        ],
    ];

    return render_admin_layout($layout);
}

function render_roles_content($data) {
    ob_start();
    $templatePath = __DIR__ . DS . 'roles' . DS . 'template.php';
    if (file_exists($templatePath)) {
        extract($data);
        include $templatePath;
    }
    return ob_get_clean();
}

function get_roles() {
    $query = '
        SELECT r.id, r.name, r.slug, r.description, r.is_system, r.permissions
        FROM roles r
        ORDER BY r.name
    ';
    $roles = db_fetch_all($query);

    foreach ($roles as &$role) {
        $permissionIds = [];
        if (!empty($role['permissions'])) {
            $permissionIds = json_decode($role['permissions'], true) ?? [];
        }
        $role['permission_ids'] = $permissionIds;
    }

    return $roles;
}

function get_all_permissions() {
    // Получаем все разрешения через хук или из конфигурации
    $permissions = [];
    if (function_exists('apply_filters')) {
        $permissions = apply_filters('admin_permissions', []);
    }

    // Базовые разрешения если хук не вернул данные
    if (empty($permissions)) {
        $permissions = [
            'admin.access' => 'Доступ до адмін-панелі',
            'admin.users.manage' => 'Управління користувачами',
            'admin.roles.manage' => 'Управління ролями',
            'admin.plugins.manage' => 'Управління плагінами',
            'admin.themes.manage' => 'Управління темами',
            'admin.settings.manage' => 'Управління налаштуваннями',
        ];
    }

    return $permissions;
}

function handle_create_role() {
    $session = function_exists('sessionManager') ? sessionManager() : null;
    $userId = $session ? (int)($session->get('admin_user_id') ?? 0) : 0;

    $hasAccess = ($userId === 0 || $userId === 1) || (function_exists('current_user_can') && current_user_can('admin.access'));
    if (!$hasAccess) {
        set_flash_message('У вас немає прав на створення ролей', 'danger');
        return;
    }

    $name = sanitize_input($_POST['name'] ?? '');
    $slug = sanitize_input($_POST['slug'] ?? '');
    $description = sanitize_input($_POST['description'] ?? '');
    $permissions = $_POST['permissions'] ?? [];

    if (empty($name) || empty($slug)) {
        set_flash_message('Заповніть всі обов\'язкові поля', 'danger');
        return;
    }

    // Проверяем существование slug
    $existing = db_fetch_one('SELECT id FROM roles WHERE slug = ?', [$slug]);
    if ($existing) {
        set_flash_message('Роль з таким slug вже існує', 'danger');
        return;
    }

    $permissionsJson = !empty($permissions) && is_array($permissions)
        ? json_encode($permissions)
        : null;

    $stmt = db_query('
        INSERT INTO roles (name, slug, description, permissions, is_system)
        VALUES (?, ?, ?, ?, 0)
    ', [$name, $slug, $description, $permissionsJson]);

    if ($stmt) {
        set_flash_message('Роль успішно створено', 'success');
        admin_redirect('roles');
    } else {
        set_flash_message('Помилка при створенні ролі', 'danger');
    }
}

function handle_update_role() {
    $session = function_exists('sessionManager') ? sessionManager() : null;
    $userId = $session ? (int)($session->get('admin_user_id') ?? 0) : 0;

    $hasAccess = ($userId === 0 || $userId === 1) || (function_exists('current_user_can') && current_user_can('admin.access'));
    if (!$hasAccess) {
        set_flash_message('У вас немає прав на редагування ролей', 'danger');
        return;
    }

    $roleId = (int)($_POST['role_id'] ?? 0);
    $name = sanitize_input($_POST['name'] ?? '');
    $description = sanitize_input($_POST['description'] ?? '');
    $permissions = $_POST['permissions'] ?? [];

    if (empty($roleId) || empty($name)) {
        set_flash_message('Невірні дані', 'danger');
        return;
    }

    // Системные роли больше не защищены от редактирования
    // Root пользователь (userId = 0) и администратор могут редактировать любые роли
    // Проверяем только базовый доступ
    if ($userId !== 0 && $userId !== 1) {
        set_flash_message('Неможливо редагувати системну роль', 'danger');
        return;
    }

    $permissionsJson = !empty($permissions) && is_array($permissions)
        ? json_encode($permissions)
        : null;

    $stmt = db_query('
        UPDATE roles
        SET name = ?, description = ?, permissions = ?
        WHERE id = ?
    ', [$name, $description, $permissionsJson, $roleId]);

    if ($stmt) {
        set_flash_message('Роль успішно оновлено', 'success');
        admin_redirect('roles');
    } else {
        set_flash_message('Помилка при оновленні ролі', 'danger');
    }
}

function handle_delete_role() {
    $session = function_exists('sessionManager') ? sessionManager() : null;
    $userId = $session ? (int)($session->get('admin_user_id') ?? 0) : 0;

    $hasAccess = ($userId === 0 || $userId === 1) || (function_exists('current_user_can') && current_user_can('admin.access'));
    if (!$hasAccess) {
        set_flash_message('У вас немає прав на видалення ролей', 'danger');
        return;
    }

    $roleId = (int)($_POST['role_id'] ?? 0);

    if (empty($roleId)) {
        set_flash_message('Невірні дані', 'danger');
        return;
    }

    // Системные роли больше не защищены от удаления
    // Root пользователь (userId = 0) и администратор могут удалять любые роли
    // Проверяем только базовый доступ
    if ($userId !== 0 && $userId !== 1) {
        set_flash_message('Неможливо видалити системну роль', 'danger');
        return;
    }

    $stmt = db_query('DELETE FROM roles WHERE id = ?', [$roleId]);

    if ($stmt) {
        set_flash_message('Роль успішно видалено', 'success');
        admin_redirect('roles');
    } else {
        set_flash_message('Помилка при видаленні ролі', 'danger');
    }
}
