<?php

declare(strict_types=1);

/**
 * Страница Login (функция вместо класса)
 * Оптимизировано для PHP 8.4+
 */

/**
 * @param array<string, mixed> $params
 * @return string|null
 */
function admin_login_page(array $params = []): ?string {
    // Если уже авторизован, редирект (только если это не POST запрос)
    if ($_SERVER['REQUEST_METHOD'] !== 'POST' && function_exists('is_admin_logged_in') && is_admin_logged_in()) {
        if (function_exists('admin_redirect')) {
            admin_redirect('dashboard');
            exit;
        }
        return null;
    }

    $error = '';

    // Обработка POST запроса
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Логируем начало обработки формы
        if (class_exists('Flowaxy\Support\Facades\Log')) {
            try {
                \Flowaxy\Support\Facades\Log::Debug('admin_login_page: Processing POST request', [
                    'username' => $_POST['username'] ?? 'NOT SET',
                    'has_password' => isset($_POST['password']),
                    'session_status' => session_status(),
                    'session_id' => session_id() ?: 'none'
                ]);
            } catch (\Throwable $e) {
                // Игнорируем ошибки логирования
            }
        }

        $error = process_login_form();

        // Логируем результат обработки формы
        if (class_exists('Flowaxy\Support\Facades\Log')) {
            try {
                \Flowaxy\Support\Facades\Log::Debug('admin_login_page: Form processing result', [
                    'error' => $error,
                    'success' => empty($error),
                    'session_status' => session_status(),
                    'session_id' => session_id() ?: 'none',
                    '$_SESSION_admin_user_id' => $_SESSION['admin_user_id'] ?? 'NOT SET',
                    'is_admin_logged_in' => function_exists('is_admin_logged_in') ? is_admin_logged_in() : 'FUNCTION NOT EXISTS'
                ]);
            } catch (\Throwable $e) {
                // Игнорируем ошибки логирования
            }
        }

        if (empty($error)) {
            // Успешный вход - редирект на dashboard
            // КРИТИЧНО: Убеждаемся, что сессия активна и данные сохранены
            if (session_status() === PHP_SESSION_ACTIVE) {
                // Явно сохраняем сессию, вызывая session_write_close() перед редиректом
                // Это гарантирует, что данные сессии будут записаны на диск
                session_write_close();
            }

            if (class_exists('Flowaxy\Support\Facades\Log')) {
                try {
                    \Flowaxy\Support\Facades\Log::Info('admin_login_page: Redirecting to dashboard', [
                        'session_status_before_close' => session_status(),
                        'session_id' => session_id() ?: 'none',
                        '$_SESSION_admin_user_id' => $_SESSION['admin_user_id'] ?? 'NOT SET',
                        'is_admin_logged_in' => function_exists('is_admin_logged_in') ? is_admin_logged_in() : 'FUNCTION NOT EXISTS',
                        'headers_sent' => headers_sent(),
                        'cookies' => $_COOKIE ?? []
                    ]);
                } catch (\Throwable $e) {
                    // Игнорируем ошибки логирования
                }
            }

            // КРИТИЧНО: Убеждаемся, что заголовки еще не отправлены перед редиректом
            if (!headers_sent()) {
                if (function_exists('admin_redirect')) {
                    admin_redirect('dashboard');
                    exit; // Убеждаемся, что выполнение прекращено
                }
                // Fallback если admin_redirect недоступен
                if (function_exists('admin_url')) {
                    header('Location: ' . admin_url('dashboard'), true, 302);
                    exit;
                }
            } else {
                // Если заголовки уже отправлены, логируем ошибку
                if (class_exists('Flowaxy\Support\Facades\Log')) {
                    try {
                        \Flowaxy\Support\Facades\Log::Error('admin_login_page: Cannot redirect, headers already sent', [
                            'session_status' => session_status(),
                            'session_id' => session_id() ?: 'none',
                            '$_SESSION_admin_user_id' => $_SESSION['admin_user_id'] ?? 'NOT SET'
                        ]);
                    } catch (\Throwable $e) {
                        // Игнорируем ошибки логирования
                    }
                }
            }
            return null;
        }
    }

    // Рендерим страницу логина
    $layout = [
        'title' => 'Вхід в адмінку - Flowaxy CMS',
        'content' => render_login_content($error),
        'template' => 'login', // Используем отдельный шаблон для логина
        'error' => $error, // Передаем ошибку в layout для шаблона
    ];

    return render_admin_layout($layout);
}

/**
 * @return string
 */
function process_login_form(): string {
    // Обработка формы входа через фреймворк
    if (!function_exists('authenticate_admin_user')) {
        return 'Система авторизации недоступна';
    }

    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';

    if (empty($username) || empty($password)) {
        return 'Заповніть всі поля';
    }

    $result = authenticate_admin_user($username, $password);

    // Логируем результат авторизации
    if (class_exists('Flowaxy\Support\Facades\Log')) {
        try {
            \Flowaxy\Support\Facades\Log::Debug('process_login_form: Authentication result', [
                'success' => $result['success'] ?? false,
                'error' => $result['error'] ?? 'NO ERROR',
                'user_id' => $result['user']['id'] ?? 'NOT SET',
                'session_status' => session_status(),
                'session_id' => session_id() ?: 'none',
                '$_SESSION_admin_user_id' => $_SESSION['admin_user_id'] ?? 'NOT SET'
            ]);
        } catch (\Throwable $e) {
            // Игнорируем ошибки логирования
        }
    }

    if (!$result || !isset($result['success']) || !$result['success']) {
        $errorMessage = $result['error'] ?? 'Невірний логін або пароль';
        error_log("process_login_form: Authentication failed, error message: {$errorMessage}");
        return $errorMessage;
    }

    // Проверяем, что сессия сохранена
    if (function_exists('is_admin_logged_in')) {
        $isLoggedIn = is_admin_logged_in();

        // Логируем проверку сессии
        if (class_exists('Flowaxy\Support\Facades\Log')) {
            try {
                \Flowaxy\Support\Facades\Log::Debug('process_login_form: Session check after auth', [
                    'is_admin_logged_in' => $isLoggedIn,
                    'session_status' => session_status(),
                    'session_id' => session_id() ?: 'none',
                    '$_SESSION_admin_user_id' => $_SESSION['admin_user_id'] ?? 'NOT SET',
                    '$_SESSION_all_keys' => array_keys($_SESSION ?? [])
                ]);
            } catch (\Throwable $e) {
                // Игнорируем ошибки логирования
            }
        }

        if (!$isLoggedIn) {
            // Логируем проблему с сессией
            if (class_exists('Flowaxy\Support\Facades\Log')) {
                try {
                    \Flowaxy\Support\Facades\Log::Warning('process_login_form: Session not saved after auth', [
                        'session_status' => session_status(),
                        'session_id' => session_id() ?: 'none',
                        '$_SESSION_admin_user_id' => $_SESSION['admin_user_id'] ?? 'NOT SET',
                        '$_SESSION_all_keys' => array_keys($_SESSION ?? [])
                    ]);
                } catch (\Throwable $e) {
                    // Игнорируем ошибки логирования
                }
            }
        }
    }

    return ''; // Успех
}

/**
 * @param string $error
 * @return string
 */
function render_login_content(string $error): string {
    ob_start();
    // Login использует отдельный layout, не template
    $layoutPath = __DIR__ . DS . '..' . DS . 'layouts' . DS . 'login.php';
    if (file_exists($layoutPath)) {
        // Передаем переменные в область видимости layout
        $csrfToken = class_exists('\Flowaxy\Support\Helpers\SecurityHelper') ? \Flowaxy\Support\Helpers\SecurityHelper::csrfToken() : (function_exists('csrf_token') ? csrf_token() : '');
        $isSecureConnection = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ||
                             (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https');
        include $layoutPath;
    } else {
        // Простой fallback
        echo '<div class="login-container">';
        if (!empty($error)) {
            echo '<div class="alert alert-danger">' . htmlspecialchars($error) . '</div>';
        }
        echo '<form method="POST">';
        echo '<input type="text" name="username" placeholder="Логін" required>';
        echo '<input type="password" name="password" placeholder="Пароль" required>';
        echo '<button type="submit">Войти</button>';
        echo '</form>';
        echo '</div>';
    }
    return ob_get_clean();
}
