<?php
/**
 * Шаблон страницы управления базой данных
 *
 * @var array $dbConfig Конфигурация БД
 * @var bool $isConnected Статус подключения
 * @var array|null $dbStats Статистика БД
 * @var array $dbTables Список таблиц
 */
?>

<div class="database-page">
    <?php if (!$isConnected): ?>
        <!-- Состояние: БД не подключена -->
        <div class="db-no-connection">
            <div class="db-no-connection-icon">
                <i class="fas fa-database"></i>
            </div>
            <h3 class="db-no-connection-title">База даних не підключена</h3>
            <p class="db-no-connection-text">
                Для роботи з базою даних необхідно налаштувати підключення.
            </p>
            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#dbConnectModal">
                <i class="fas fa-plug me-2"></i>
                Підключити базу даних
            </button>
        </div>
    <?php else: ?>
        <!-- Состояние: БД подключена -->
        <div class="mb-3 d-flex justify-content-between align-items-center">
            <div>
                <span class="db-status-badge db-status-connected">
                    <i class="fas fa-check-circle me-1"></i>
                    Підключено
                </span>
            </div>
            <div>
                <button type="button" class="btn btn-sm btn-outline-secondary me-2" data-bs-toggle="modal" data-bs-target="#dbConnectModal">
                    <i class="fas fa-cog me-1"></i>
                    Налаштування
                </button>
                <form method="POST" action="<?= admin_url('database') ?>" style="display: inline;" onsubmit="return confirm('Ви впевнені, що хочете відключити базу даних?');">
                    <?= csrf_field() ?>
                    <input type="hidden" name="action" value="disconnect">
                    <button type="submit" class="btn btn-sm btn-outline-danger">
                        <i class="fas fa-unlink me-1"></i>
                        Відключити
                    </button>
                </form>
            </div>
        </div>

        <?php if ($dbStats): ?>
            <!-- Статистика БД -->
            <div class="db-stats-section mb-3">
                <?php
                $componentsPath = __DIR__ . DS . '..' . DS . '..' . DS . 'components' . DS;
                $cards = [
                    [
                        'title' => 'Запитів виконано',
                        'value' => number_format($dbStats['query_count'] ?? 0, 0, ',', ' '),
                        'icon' => 'database',
                        'color' => 'primary'
                    ],
                    [
                        'title' => 'Середній час запиту',
                        'value' => number_format($dbStats['average_query_time'] ?? 0, 4, ',', ' ') . 's',
                        'icon' => 'clock',
                        'color' => 'info'
                    ],
                    [
                        'title' => 'Помилок',
                        'value' => ($dbStats['error_count'] ?? 0) > 0
                            ? '<span class="text-danger">' . number_format($dbStats['error_count'], 0, ',', ' ') . '</span>'
                            : '<span class="text-muted">0</span>',
                        'icon' => 'exclamation-triangle',
                        'color' => ($dbStats['error_count'] ?? 0) > 0 ? 'danger' : 'secondary',
                        'valueClass' => 'h5'
                    ],
                    [
                        'title' => 'Повільних запитів',
                        'value' => ($dbStats['slow_queries'] ?? 0) > 0
                            ? '<span class="text-warning">' . number_format($dbStats['slow_queries'], 0, ',', ' ') . '</span>'
                            : '<span class="text-muted">0</span>',
                        'icon' => 'hourglass-half',
                        'color' => ($dbStats['slow_queries'] ?? 0) > 0 ? 'warning' : 'secondary',
                        'valueClass' => 'h5'
                    ],
                ];

                // Добавляем дополнительные карточки, если данные доступны
                if (isset($dbStats['mysql_uptime'])) {
                    $cards[] = [
                        'title' => 'Uptime (сек)',
                        'value' => number_format($dbStats['mysql_uptime'], 0, ',', ' '),
                        'icon' => 'server',
                        'color' => 'success'
                    ];
                }

                if (isset($dbStats['mysql_threads_connected'])) {
                    $cards[] = [
                        'title' => 'Підключень',
                        'value' => number_format($dbStats['mysql_threads_connected'], 0, ',', ' '),
                        'icon' => 'plug',
                        'color' => 'info'
                    ];
                }

                // Используем компонент stats-cards
                // Устанавливаем columnsPerRow для отображения всех карточек в одну строку
                $columnsPerRow = count($cards);
                $fixedWidth = true;
                $statsCardsPath = $componentsPath . 'layout' . DS . 'stats-cards.php';
                if (file_exists($statsCardsPath)) {
                    include $statsCardsPath;
                }
                ?>
            </div>
        <?php endif; ?>

        <!-- Таблицы БД -->
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0"><i class="fas fa-table me-2"></i>Таблиці (<?= count($dbTables) ?>)</h5>
                <button type="button" class="btn btn-sm btn-outline-primary" onclick="DB.refreshTables()">
                    <i class="fas fa-sync-alt"></i>
                </button>
            </div>
            <div class="card-body p-0">
                <?php if (empty($dbTables)): ?>
                    <div class="text-center py-4 text-muted">
                        <i class="fas fa-inbox fa-2x mb-2"></i>
                        <p>Таблиці не знайдено</p>
                    </div>
                <?php else: ?>
                    <div class="db-tables-table">
                        <table class="table table-hover mb-0">
                            <thead>
                                <tr>
                                    <th>Назва</th>
                                    <th>Рядків</th>
                                    <th>Розмір (MB)</th>
                                    <th>Collation</th>
                                    <th>Engine</th>
                                    <?php if ($hasAdminAccess ?? false): ?>
                                        <th>Дії</th>
                                    <?php endif; ?>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($dbTables as $table): ?>
                                    <tr>
                                        <td><strong><?= htmlspecialchars($table['name']) ?></strong></td>
                                        <td><?= number_format($table['rows'] ?? 0) ?></td>
                                        <td><?= number_format($table['size_mb'] ?? 0, 2) ?></td>
                                        <td><code><?= htmlspecialchars($table['collation'] ?? '-') ?></code></td>
                                        <td><span class="badge bg-secondary"><?= htmlspecialchars($table['engine'] ?? '-') ?></span></td>
                                        <?php if ($hasAdminAccess ?? false): ?>
                                            <td>
                                                <div class="btn-group btn-group-sm" role="group">
                                                    <button type="button" class="btn btn-outline-primary"
                                                            onclick="DB.optimizeTable('<?= htmlspecialchars($table['name'], ENT_QUOTES) ?>')"
                                                            title="Оптимізувати таблицю">
                                                        <i class="fas fa-compress-alt"></i>
                                                    </button>
                                                    <button type="button" class="btn btn-outline-danger"
                                                            onclick="DB.truncateTable('<?= htmlspecialchars($table['name'], ENT_QUOTES) ?>')"
                                                            title="Очистити таблицю (видалити всі дані)">
                                                        <i class="fas fa-trash-alt"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        <?php endif; ?>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    <?php endif; ?>
</div>

<!-- Модальное окно подключения к БД -->
<div class="modal fade" id="dbConnectModal" tabindex="-1" aria-labelledby="dbConnectModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="dbConnectModalLabel">
                    <i class="fas fa-database me-2"></i>
                    Налаштування підключення до бази даних
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="dbConnectForm" method="POST" action="<?= admin_url('database') ?>">
                <?= csrf_field() ?>
                <input type="hidden" name="action" value="connect">
                <div class="modal-body">
                    <div id="dbTestResult" class="alert d-none mb-3"></div>

                    <div class="mb-3">
                        <label for="dbStatus" class="form-label">
                            <input type="checkbox" name="status" id="dbStatus" value="1" <?= ($dbConfig['status'] === 'true' || $dbConfig['status'] === true || $dbConfig['status'] === '1') ? 'checked' : '' ?>>
                            Увімкнути підключення
                        </label>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="dbHost" class="form-label">Хост <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="dbHost" name="host" value="<?= htmlspecialchars($dbConfig['host']) ?>" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="dbPort" class="form-label">Порт</label>
                            <input type="number" class="form-control" id="dbPort" name="port" value="<?= htmlspecialchars($dbConfig['port']) ?>" min="1" max="65535">
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="dbName" class="form-label">Назва бази даних</label>
                        <input type="text" class="form-control" id="dbName" name="name" value="<?= htmlspecialchars($dbConfig['name']) ?>">
                        <small class="form-text text-muted">Якщо база не існує, вона не буде створена автоматично</small>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="dbUser" class="form-label">Користувач <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="dbUser" name="user" value="<?= htmlspecialchars($dbConfig['user']) ?>" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="dbPass" class="form-label">Пароль</label>
                            <input type="password" class="form-control" id="dbPass" name="pass" value="<?= htmlspecialchars($dbConfig['pass']) ?>">
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="dbCharset" class="form-label">Кодування</label>
                        <select class="form-select" id="dbCharset" name="charset">
                            <option value="utf8mb4" <?= $dbConfig['charset'] === 'utf8mb4' ? 'selected' : '' ?>>utf8mb4 (рекомендовано)</option>
                            <option value="utf8" <?= $dbConfig['charset'] === 'utf8' ? 'selected' : '' ?>>utf8</option>
                            <option value="latin1" <?= $dbConfig['charset'] === 'latin1' ? 'selected' : '' ?>>latin1</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Скасувати</button>
                    <button type="button" class="btn btn-outline-primary" onclick="DB.testConnection()">
                        <i class="fas fa-vial me-2"></i>
                        Перевірити підключення
                    </button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save me-2"></i>
                        Зберегти
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
const DB = {
    testConnection: function() {
        const form = document.getElementById('dbConnectForm');
        const resultDiv = document.getElementById('dbTestResult');
        const testBtn = event.target;
        const originalText = testBtn.innerHTML;

        testBtn.disabled = true;
        testBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Перевірка...';
        resultDiv.classList.add('d-none');

        const formData = new FormData(form);
        formData.append('action', 'test_connection');
        formData.append('_token', formData.get('_token'));

        fetch('<?= admin_url('database') ?>', {
            method: 'POST',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            resultDiv.classList.remove('d-none');
            if (data.success) {
                resultDiv.className = 'alert alert-success mb-3';
                resultDiv.innerHTML = '<i class="fas fa-check-circle me-2"></i>' + data.message;
            } else {
                resultDiv.className = 'alert alert-danger mb-3';
                resultDiv.innerHTML = '<i class="fas fa-times-circle me-2"></i>' + data.message;
            }
        })
        .catch(error => {
            resultDiv.classList.remove('d-none');
            resultDiv.className = 'alert alert-danger mb-3';
            resultDiv.innerHTML = '<i class="fas fa-times-circle me-2"></i>Помилка: ' + error.message;
        })
        .finally(() => {
            testBtn.disabled = false;
            testBtn.innerHTML = originalText;
        });
    },

    refreshTables: function() {
        const btn = event.target.closest('button');
        const originalText = btn.innerHTML;
        btn.disabled = true;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i>Оновлення...';

        const formData = new FormData();
        formData.append('action', 'get_tables');
        formData.append('_token', '<?= csrf_token() ?>');

        fetch('<?= admin_url('database') ?>', {
            method: 'POST',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert('Помилка оновлення: ' + (data.message || 'Невідома помилка'));
            }
        })
        .catch(error => {
            alert('Помилка: ' + error.message);
        })
        .finally(() => {
            btn.disabled = false;
            btn.innerHTML = originalText;
        });
    },

    optimizeTable: function(tableName) {
        if (!confirm('Ви впевнені, що хочете оптимізувати таблицю "' + tableName + '"?')) {
            return;
        }

        const formData = new FormData();
        formData.append('action', 'optimize_table');
        formData.append('table', tableName);
        formData.append('_token', '<?= csrf_token() ?>');

        fetch('<?= admin_url('database') ?>', {
            method: 'POST',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Таблиця "' + tableName + '" успішно оптимізована');
                location.reload();
            } else {
                alert('Помилка оптимізації: ' + (data.message || 'Невідома помилка'));
            }
        })
        .catch(error => {
            alert('Помилка: ' + error.message);
        });
    },

    truncateTable: function(tableName) {
        if (!confirm('УВАГА! Ви впевнені, що хочете очистити таблицю "' + tableName + '"?\n\nВсі дані будуть видалені безповоротно!')) {
            return;
        }

        if (!confirm('Останнє підтвердження: видалити всі дані з таблиці "' + tableName + '"?')) {
            return;
        }

        const formData = new FormData();
        formData.append('action', 'truncate_table');
        formData.append('table', tableName);
        formData.append('_token', '<?= csrf_token() ?>');

        fetch('<?= admin_url('database') ?>', {
            method: 'POST',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Таблиця "' + tableName + '" успішно очищена');
                location.reload();
            } else {
                alert('Помилка очищення: ' + (data.message || 'Невідома помилка'));
            }
        })
        .catch(error => {
            alert('Помилка: ' + error.message);
        });
    }
};
</script>
