<?php

declare(strict_types=1);

/**
 * Страница очистки кеша (AJAX) (функция вместо класса)
 * Оптимизировано для PHP 8.4+
 */

/**
 * @return never
 */
function admin_cache_clear_page(): never {
    // Проверка авторизации
    if (!function_exists('current_user_can') || !current_user_can('admin.access')) {
        send_json_response(['success' => false, 'error' => 'Доступ запрещен'], 403);
    }

    // Проверка AJAX запроса
    $isAjax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) &&
              strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';

    if (!$isAjax) {
        if (function_exists('admin_redirect')) {
            admin_redirect('dashboard');
        } else {
            // Fallback если admin_redirect не доступен
            header('Location: ' . admin_url('dashboard'));
            exit;
        }
    }

    $action = sanitize_input($_POST['cache_action'] ?? '');

    if (empty($action)) {
        send_json_response(['success' => false, 'error' => 'Дія не вказана'], 400);
    }

    try {
        if ($action === 'clear_all') {
            clear_all_cache();
        } elseif ($action === 'clear_expired') {
            clear_expired_cache();
        } else {
            send_json_response(['success' => false, 'error' => 'Невідома дія'], 400);
        }
    } catch (\Exception $e) {
        if (function_exists('logger')) {
            logger()->logError('Cache clear error: ' . $e->getMessage(), ['exception' => $e]);
        }
        send_json_response(['success' => false, 'error' => 'Помилка при очищенні кешу'], 500);
    }
}

/**
 * @return never
 */
function clear_all_cache(): never {
    if (function_exists('cache')) {
        $cache = cache();
        $result = $cache->clear();

        if ($result) {
            send_json_response(['success' => true, 'message' => 'Весь кеш успішно очищено'], 200);
        } else {
            send_json_response(['success' => false, 'error' => 'Помилка при очищенні кешу'], 500);
        }
    } else {
        send_json_response(['success' => false, 'error' => 'Кеш недоступен'], 500);
    }
}

/**
 * @return never
 */
function clear_expired_cache(): never {
    if (function_exists('cache')) {
        $cache = cache();
        $cleaned = $cache->cleanup();

        send_json_response([
            'success' => true,
            'message' => 'Прострочений кеш очищено. Видалено файлів: ' . $cleaned,
            'cleaned_count' => $cleaned
        ], 200);
    } else {
        send_json_response(['success' => false, 'error' => 'Кеш недоступен'], 500);
    }
}
