<?php

declare(strict_types=1);

/**
 * Вспомогательные функции для админки
 * Оптимизировано для PHP 8.4+
 */

/**
 * Рендеринг layout админки
 * Переопределяет базовую реализацию из AdminHelper.php
 *
 * @param array{title?: string, content?: string, template?: string} $config
 * @return string
 */
if (!function_exists('render_admin_layout')) {
    function render_admin_layout(array $config): string
    {
        $title = $config['title'] ?? 'Админ-панель';
        $content = $config['content'] ?? '';
        $template = $config['template'] ?? 'layout';

        // Если используется отдельный шаблон (например, login)
        if ($template === 'login') {
            $layoutPath = __DIR__ . DS . '..' . DS . 'layouts' . DS . 'login.php';
            if (file_exists($layoutPath)) {
                // Передаем переменные в layout
                $error = $config['error'] ?? '';
                $csrfToken = class_exists('\Flowaxy\Support\Helpers\SecurityHelper') ? \Flowaxy\Support\Helpers\SecurityHelper::csrfToken() : (function_exists('csrf_token') ? csrf_token() : '');
                ob_start();
                include $layoutPath;
                return ob_get_clean();
            }
        }

        // Обычный layout с Header, Sidebar, Footer
        ob_start();
        $layoutPath = __DIR__ . DS . '..' . DS . 'layouts' . DS . 'main.php';
        if (file_exists($layoutPath)) {
            // Извлекаем переменные из конфига в область видимости шаблона
            // EXTR_SKIP - не перезаписывает существующие переменные
            extract($config, EXTR_SKIP);

            // Убеждаемся, что основные переменные определены (на случай если их нет в config)
            if (!isset($pageBreadcrumbs)) {
                $pageBreadcrumbs = $config['pageBreadcrumbs'] ?? [];
            }
            if (!isset($pageHeaderTitle)) {
                $pageHeaderTitle = $config['pageHeaderTitle'] ?? '';
            }
            if (!isset($pageHeaderDescription)) {
                $pageHeaderDescription = $config['pageHeaderDescription'] ?? '';
            }
            if (!isset($pageHeaderIcon)) {
                $pageHeaderIcon = $config['pageHeaderIcon'] ?? '';
            }
            if (!isset($pageHeaderButtons)) {
                $pageHeaderButtons = $config['pageHeaderButtons'] ?? '';
            }

            include $layoutPath;
        } else {
            // Простой fallback
            echo '<!DOCTYPE html><html><head><title>' . htmlspecialchars($title) . '</title></head><body>';
            echo $content;
            echo '</body></html>';
        }
        return ob_get_clean();
    }
}

/**
 * Редирект в админке
 * Переопределяет базовую реализацию из AdminHelper.php
 *
 * @param string $page
 * @return never
 */
if (!function_exists('admin_redirect')) {
    function admin_redirect(string $page): never
    {
        $url = admin_url($page);
        header('Location: ' . $url);
        exit;
    }
}

/**
 * Получить URL админки
 * Использует функцию из AdminHelper если доступна
 */
if (!function_exists('admin_url')) {
    /**
     * @param string $page
     * @return string
     */
    function admin_url(string $page = ''): string
    {
        // Функция уже определена в AdminHelper.php
        // Если она не загружена, используем fallback
        if (function_exists('\admin_url')) {
            return \admin_url($page);
        }
        return '/admin/' . ($page !== '' ? '/' . ltrim($page, '/') : '');
    }
}

/**
 * Проверка авторизации в админке
 * Использует функцию из AdminHelper если доступна
 */
if (!function_exists('is_admin_logged_in')) {
    /**
     * @return bool
     */
    function is_admin_logged_in(): bool
    {
        // Функция уже определена в AdminHelper.php
        // Если она не загружена, используем fallback
        if (function_exists('\is_admin_logged_in')) {
            return \is_admin_logged_in();
        }
        return false;
    }
}

/**
 * Получение менеджера сессии
 * Возвращает экземпляр SessionManager для работы с сессией
 *
 * @return \Flowaxy\Support\Managers\SessionManager|null
 */
if (!function_exists('sessionManager')) {
    function sessionManager(): ?\Flowaxy\Support\Managers\SessionManager
    {
        // Пробуем через фасад Session
        if (class_exists('\Flowaxy\Support\Facades\Session')) {
            try {
                return \Flowaxy\Support\Facades\Session::manager();
            } catch (\Throwable $e) {
                // Фасад не доступен, пробуем fallback
            }
        }

        // Fallback: используем SessionManager напрямую
        if (class_exists('\Flowaxy\Support\Managers\SessionManager')) {
            try {
                return \Flowaxy\Support\Managers\SessionManager::getInstance();
            } catch (\Throwable $e) {
                // SessionManager не доступен
            }
        }

        return null;
    }
}

/**
 * Авторизация пользователя админки
 * Функция определена в flowaxy/Support/Helpers/AdminHelper.php
 * Здесь не определяем, чтобы избежать конфликтов
 */
// Функция authenticate_admin_user определена в flowaxy/Support/Helpers/AdminHelper.php
// Не определяем здесь, чтобы избежать конфликтов

/**
 * Получение соединения с БД
 * Использует DatabaseHelper из фреймворка
 *
 * @return \PDO|null
 */
function get_db_connection(): ?\PDO
{
    if (class_exists('\Flowaxy\Support\Helpers\DatabaseHelper')) {
        return \Flowaxy\Support\Helpers\DatabaseHelper::getConnection(false);
    }
    return null;
}

/**
 * Выполнение SQL запроса
 *
 * @param string $query
 * @param array<int|string, mixed> $params
 * @return \PDOStatement|false
 */
function db_query(string $query, array $params = []): \PDOStatement|false
{
    $db = get_db_connection();
    if ($db === null) {
        return false;
    }

    try {
        $stmt = $db->prepare($query);
        $stmt->execute($params);
        return $stmt;
    } catch (\Exception $e) {
        if (function_exists('logger')) {
            logger()->logError('DB query error: ' . $e->getMessage(), ['query' => $query, 'params' => $params]);
        }
        return false;
    }
}

/**
 * Применение фильтра (hook) - алиас для hook_apply и applyFilter
 * Используется в шаблонах для совместимости с WordPress-style API
 *
 * @param string $hookName
 * @param mixed $data
 * @param mixed ...$args
 * @return mixed
 */
if (!function_exists('add_filter')) {
    /**
     * Добавление фильтра (filter hook)
     * @param string $hookName Название хука
     * @param callable $callback Callback функция
     * @param int $priority Приоритет (по умолчанию 10)
     * @return void
     */
    function add_filter(string $hookName, callable $callback, int $priority = 10): void
    {
        // Пробуем использовать Filter::add если доступен
        if (class_exists('\Flowaxy\Core\Hooks\Filter')) {
            \Flowaxy\Core\Hooks\Filter::add($hookName, $callback, $priority);
            return;
        }

        // Пробуем использовать addFilter если доступен
        if (function_exists('addFilter')) {
            addFilter($hookName, $callback, $priority);
            return;
        }

        // Пробуем использовать Hooks facade если доступен
        // Если Hooks facade доступен, используем его
        // (метод addFilter існує в PluginManager helpers, але може бути недоступний глобально)
        // В іншому випадку просто ігноруємо - фільтр не буде зареєстрований.
    }
}

if (!function_exists('apply_filters')) {
    function apply_filters(string $hookName, $data = null, ...$args)
    {
        // Основний шлях: Flowaxy filters
        if (class_exists('\Flowaxy\Core\Hooks\Filter')) {
            // Приводимо args до context (перший масив або весь список)
            $context = [];
            if (!empty($args)) {
                $context = is_array($args[0]) ? $args[0] : $args;
            }
            return \Flowaxy\Core\Hooks\Filter::apply($hookName, $data, is_array($context) ? $context : []);
        }

        // Пробуем использовать Hooks facade если доступен
        if (class_exists('\Flowaxy\Support\Facades\Hooks')) {
            $context = [];
            if (!empty($args)) {
                $context = is_array($args[0]) ? $args[0] : $args;
            }
            try {
                return \Flowaxy\Support\Facades\Hooks::apply($hookName, $data, $context);
            } catch (\Throwable $e) {
                // Игнорируем ошибки, если Hooks недоступен
            }
        }

        // Fallback: возвращаем данные без изменений
        return $data;
    }
}

/**
 * Включение компонента - обертка над ComponentHelper::includeComponent()
 * Используется в шаблонах для удобства
 *
 * @param string $componentName
 * @param array<string, mixed> $data
 * @return void
 */
if (!function_exists('includeComponent')) {
    function includeComponent(string $componentName, array $data = []): void
    {
        // Пробуем использовать ComponentHelper если доступен
        if (class_exists('\Flowaxy\Support\Helpers\ComponentHelper')) {
            \Flowaxy\Support\Helpers\ComponentHelper::includeComponent($componentName, $data);
            return;
        }

        // Fallback: пробуем загрузить компонент напрямую
        if (class_exists('\Flowaxy\Core\System\PathResolver')) {
            $componentPath = \Flowaxy\Core\System\PathResolver::contentAdmin() .
                (defined('DS') ? DS : DIRECTORY_SEPARATOR) .
                'components' .
                (defined('DS') ? DS : DIRECTORY_SEPARATOR) .
                $componentName . '.php';

            if (file_exists($componentPath) && is_readable($componentPath)) {
                extract($data);
                include $componentPath;
                return;
            }
        }

        // Если компонент не найден, просто игнорируем (не показываем ошибку пользователю)
    }
}

/**
 * Получение всех записей из запроса
 *
 * @param string $query
 * @param array<int|string, mixed> $params
 * @return array<int, array<string, mixed>>
 */
function db_fetch_all(string $query, array $params = []): array
{
    $stmt = db_query($query, $params);
    if ($stmt === false) {
        return [];
    }
    return $stmt->fetchAll(\PDO::FETCH_ASSOC) ?: [];
}

/**
 * Получение одной записи из запроса
 *
 * @param string $query
 * @param array<int|string, mixed> $params
 * @return array<string, mixed>|null
 */
function db_fetch_one(string $query, array $params = []): ?array
{
    $stmt = db_query($query, $params);
    if ($stmt === false) {
        return null;
    }
    return $stmt->fetch(\PDO::FETCH_ASSOC) ?: null;
}

/**
 * Вспомогательная функция для санитизации ввода
 */
if (!function_exists('sanitize_input')) {
    /**
     * @param mixed $value
     * @return string
     */
    function sanitize_input(mixed $value): string
    {
        if (class_exists('\Flowaxy\Support\Helpers\SecurityHelper')) {
            return \Flowaxy\Support\Helpers\SecurityHelper::sanitizeInput($value);
        }
        return htmlspecialchars(trim((string)$value), ENT_QUOTES, 'UTF-8');
    }
}

/**
 * Установка flash сообщения
 */
if (!function_exists('set_flash_message')) {
    /**
     * @param string $message
     * @param string $type
     * @return void
     */
    function set_flash_message(string $message, string $type = 'info'): void
    {
        $session = function_exists('sessionManager') ? sessionManager() : null;
        if ($session !== null) {
            // Используем setFlash для установки flash-сообщения
            if (method_exists($session, 'setFlash')) {
                $session->setFlash('admin_message', $message);
                $session->setFlash('admin_message_type', $type);
            } else {
                // Fallback: используем set
                $session->set('admin_message', $message);
                $session->set('admin_message_type', $type);
            }
        }
    }
}

/**
 * Отправка JSON ответа
 */
if (!function_exists('send_json_response')) {
    /**
     * @param mixed $data
     * @param int $code
     * @return never
     */
    function send_json_response(mixed $data, int $code = 200): never
    {
        http_response_code($code);
        header('Content-Type: application/json; charset=UTF-8');
        echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_THROW_ON_ERROR);
        exit;
    }
}

/**
 * Получение CSRF токена
 */
if (!function_exists('csrf_token')) {
    /**
     * @return string
     */
    function csrf_token(): string
    {
        if (class_exists('\Flowaxy\Support\Helpers\SecurityHelper')) {
            return \Flowaxy\Support\Helpers\SecurityHelper::csrfToken();
        }
        return '';
    }
}

/**
 * Получить CSRF поле для формы
 */
if (!function_exists('csrf_field')) {
    /**
     * @return string
     */
    function csrf_field(): string
    {
        $token = csrf_token();
        return '<input type="hidden" name="csrf_token" value="' . htmlspecialchars($token) . '">';
    }
}

/**
 * Проверка CSRF токена
 */
if (!function_exists('verify_csrf_token')) {
    /**
     * @param string|null $token
     * @return bool
     */
    function verify_csrf_token(?string $token = null): bool
    {
        if (class_exists('\Flowaxy\Support\Helpers\SecurityHelper')) {
            return \Flowaxy\Support\Helpers\SecurityHelper::verifyCsrfToken($token);
        }
        return false;
    }
}

/**
 * Получение данных через хуки
 * Использует ContentHelper если доступен
 *
 * @param string $hook
 * @param mixed $default
 * @return mixed
 */
if (!function_exists('get_content_data')) {
    function get_content_data(string $hook, mixed $default = []): mixed
    {
        // ВАЖЛИВО: не викликаємо get_content_data() всередині себе (інакше рекурсія).
        if (class_exists('\Flowaxy\Core\Hooks\Filter')) {
            return \Flowaxy\Core\Hooks\Filter::apply($hook, $default, []);
        }
        if (function_exists('apply_filters')) {
            return apply_filters($hook, $default);
        }
        return $default;
    }
}

/**
 * Генерация URL для ассетов админки
 * Использует UrlHelper::adminAsset() для правильной генерации URL с версионированием
 *
 * @param string $path Путь к ассету относительно content/admin/resources/
 * @return string Полный URL к ассету
 */
if (!function_exists('asset')) {
    function asset(string $path): string
    {
        // Используем UrlHelper::adminAsset() если доступен
        if (class_exists('\Flowaxy\Support\Helpers\UrlHelper')) {
            return \Flowaxy\Support\Helpers\UrlHelper::adminAsset($path);
        }

        // Fallback: используем ComponentHelper::asset() если доступен
        if (class_exists('\Flowaxy\Support\Helpers\ComponentHelper')) {
            return \Flowaxy\Support\Helpers\ComponentHelper::asset($path);
        }

        // Простой fallback
        return '/content/admin/resources/' . ltrim($path, '/');
    }
}

/**
 * Получение URL превью для плагинов/тем с fallback на системную заглушку.
 *
 * @param string $type 'plugin' | 'theme'
 * @param string $slug slug плагина/темы
 * @return array{url: string, is_placeholder: bool}
 */
if (!function_exists('admin_package_preview')) {
    function admin_package_preview(string $type, string $slug): array
    {
        $type = strtolower(trim($type));
        $slug = trim($slug);

        $isPlugin = ($type === 'plugin');
        $folder = $isPlugin ? 'plugins' : 'themes';
        // Используем дефолтные иконки для плагинов и тем
        $fallbackAsset = $isPlugin ? 'images/system/plugin.png' : 'images/system/theme.png';

        // Безопасный slug (защита от path traversal)
        $safeSlug = preg_replace('/[^a-z0-9\-_]/i', '', $slug ?? '') ?? '';

        if ($safeSlug !== '') {
            $ds = defined('DS') ? DS : DIRECTORY_SEPARATOR;

            // Абсолютный путь до content/
            $contentDir = null;
            if (class_exists('\Flowaxy\Core\System\PathResolver')) {
                try {
                    $contentDir = \Flowaxy\Core\System\PathResolver::content();
                } catch (\Throwable $e) {
                    $contentDir = null;
                }
            }
            if (empty($contentDir)) {
                // __DIR__ = content/admin/includes → dirname(..., 2) = content
                $contentDir = dirname(__DIR__, 2);
            }

            $previewFile = $contentDir . $ds . $folder . $ds . $safeSlug . $ds . 'preview.png';
            if (is_string($previewFile) && file_exists($previewFile)) {
                // Публичный URL к content/{plugins|themes}/{slug}/preview.png
                if (class_exists('\Flowaxy\Support\Helpers\UrlHelper')) {
                    return [
                        'url' => \Flowaxy\Support\Helpers\UrlHelper::protocolRelative('content/' . $folder . '/' . $safeSlug . '/preview.png'),
                        'is_placeholder' => false,
                    ];
                }

                return [
                    'url' => '/content/' . $folder . '/' . $safeSlug . '/preview.png',
                    'is_placeholder' => false,
                ];
            }
        }

        return [
            'url' => function_exists('asset') ? asset($fallbackAsset) : ('/content/admin/resources/' . $fallbackAsset),
            'is_placeholder' => true,
        ];
    }
}

/**
 * Проверка доступности базы данных
 * @return bool
 */
if (!function_exists('is_database_available')) {
    function is_database_available(): bool
    {
        // Проверяем наличие констант БД
        if (!defined('DB_HOST') || empty(DB_HOST) || !defined('DB_NAME') || empty(DB_NAME)) {
            return false;
        }

        // Проверяем доступность через DatabaseHelper
        if (class_exists('\Flowaxy\Support\Helpers\DatabaseHelper')) {
            return \Flowaxy\Support\Helpers\DatabaseHelper::isAvailable(false);
        }

        return false;
    }
}

/**
 * Получение менеджера плагинов
 * Возвращает экземпляр PluginManager для работы с плагинами
 *
 * @return \Flowaxy\Support\Managers\PluginManager|null
 */
if (!function_exists('pluginManager')) {
    function pluginManager(): ?\Flowaxy\Support\Managers\PluginManager
    {
        // Сначала убеждаемся, что класс PluginManager загружен
        if (!class_exists('\Flowaxy\Support\Managers\PluginManager', false)) {
            // Пробуем использовать PathResolver для определения пути
            if (class_exists('\Flowaxy\Core\System\PathResolver')) {
                $flowaxyDir = \Flowaxy\Core\System\PathResolver::flowaxy();
                $ds = defined('DS') ? DS : DIRECTORY_SEPARATOR;
                $pluginManagerFile = $flowaxyDir . $ds . 'Support' . $ds . 'Managers' . $ds . 'PluginManager.php';

                if (file_exists($pluginManagerFile) && is_readable($pluginManagerFile)) {
                    require_once $pluginManagerFile;
                } else {
                    @error_log("pluginManager: PluginManager.php file not found at: " . $pluginManagerFile);
                    return null;
                }
            } else {
                // Fallback: пробуем определить путь вручную
                $baseDir = dirname(__DIR__, 3);
                $ds = defined('DS') ? DS : DIRECTORY_SEPARATOR;
                $pluginManagerFile = $baseDir . $ds . 'flowaxy' . $ds . 'Support' . $ds . 'Managers' . $ds . 'PluginManager.php';

                if (file_exists($pluginManagerFile) && is_readable($pluginManagerFile)) {
                    require_once $pluginManagerFile;
                } else {
                    @error_log("pluginManager: PluginManager.php file not found at: " . $pluginManagerFile);
                    return null;
                }
            }
        }

        // Используем ManagerHelper если доступен
        if (class_exists('\Flowaxy\Support\Helpers\ManagerHelper')) {
            try {
                $manager = \Flowaxy\Support\Helpers\ManagerHelper::pluginManager();
                if ($manager !== null) {
                    return $manager;
                }
            } catch (\Throwable $e) {
                @error_log("pluginManager: ManagerHelper error - " . $e->getMessage());
            }
        }

        // Fallback: используем PluginManager напрямую
        if (class_exists('\Flowaxy\Support\Managers\PluginManager')) {
            try {
                return \Flowaxy\Support\Managers\PluginManager::getInstance();
            } catch (\Throwable $e) {
                @error_log("pluginManager: PluginManager::getInstance() error - " . $e->getMessage());
            }
        }

        @error_log("pluginManager: PluginManager class not found after loading");
        return null;
    }
}

/**
 * Получение менеджера тем
 * Возвращает экземпляр ThemeManager для работы с темами
 *
 * @return \Flowaxy\Support\Managers\ThemeManager|null
 */
if (!function_exists('themeManager')) {
    function themeManager(): ?\Flowaxy\Support\Managers\ThemeManager
    {
        // Сначала убеждаемся, что класс ThemeManager загружен
        if (!class_exists('\Flowaxy\Support\Managers\ThemeManager', false)) {
            // Пробуем использовать PathResolver для определения пути
            if (class_exists('\Flowaxy\Core\System\PathResolver')) {
                $flowaxyDir = \Flowaxy\Core\System\PathResolver::flowaxy();
                $ds = defined('DS') ? DS : DIRECTORY_SEPARATOR;
                $themeManagerFile = $flowaxyDir . $ds . 'Support' . $ds . 'Managers' . $ds . 'ThemeManager.php';

                if (file_exists($themeManagerFile) && is_readable($themeManagerFile)) {
                    require_once $themeManagerFile;
                } else {
                    @error_log("themeManager: ThemeManager.php file not found at: " . $themeManagerFile);
                    return null;
                }
            } else {
                // Fallback: пробуем определить путь вручную
                $baseDir = dirname(__DIR__, 3);
                $ds = defined('DS') ? DS : DIRECTORY_SEPARATOR;
                $themeManagerFile = $baseDir . $ds . 'flowaxy' . $ds . 'Support' . $ds . 'Managers' . $ds . 'ThemeManager.php';

                if (file_exists($themeManagerFile) && is_readable($themeManagerFile)) {
                    require_once $themeManagerFile;
                } else {
                    @error_log("themeManager: ThemeManager.php file not found at: " . $themeManagerFile);
                    return null;
                }
            }
        }

        // Используем ManagerHelper если доступен
        if (class_exists('\Flowaxy\Support\Helpers\ManagerHelper')) {
            try {
                $manager = \Flowaxy\Support\Helpers\ManagerHelper::themeManager();
                if ($manager !== null) {
                    return $manager;
                }
            } catch (\Throwable $e) {
                @error_log("themeManager: ManagerHelper error - " . $e->getMessage());
            }
        }

        // Fallback: используем ThemeManager напрямую
        if (class_exists('\Flowaxy\Support\Managers\ThemeManager')) {
            try {
                return \Flowaxy\Support\Managers\ThemeManager::getInstance();
            } catch (\Throwable $e) {
                @error_log("themeManager: ThemeManager::getInstance() error - " . $e->getMessage());
            }
        }

        @error_log("themeManager: ThemeManager class not found after loading");
        return null;
    }
}

/**
 * @deprecated Функція register_public_route() перенесена в flowaxy/Support/Helpers/RouteHelper.php
 * Використовується плагінами та темами, тому не повинна бути в адмінці.
 * Функція завантажується автоматично через HelperFunctionsLoader.
 */

/**
 * Рендеринг сообщения о необходимости настройки БД
 * @param string $moduleName Название модуля
 * @return string
 */
if (!function_exists('render_database_required_message')) {
    function render_database_required_message(string $moduleName): string
    {
        ob_start();
?>
        <div class="row">
            <div class="col-md-12">
                <div class="alert alert-warning" role="alert">
                    <h4 class="alert-heading">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        База даних не налаштована
                    </h4>
                    <p>
                        Для роботи модуля <strong><?= htmlspecialchars($moduleName) ?></strong> необхідно налаштувати підключення до бази даних.
                    </p>
                    <hr>
                    <p class="mb-0">
                        <strong>Щоб налаштувати базу даних:</strong>
                    </p>
                    <ol class="mt-2">
                        <li>Перейдіть до <a href="<?= admin_url('settings') ?>" class="alert-link">Налаштувань</a></li>
                        <li>Виберіть тип бази даних (MySQL, PostgreSQL, SQLite)</li>
                        <li>Введіть параметри підключення</li>
                        <li>Збережіть налаштування</li>
                    </ol>
                    <p class="mt-3 mb-0">
                        <small class="text-muted">
                            <i class="fas fa-info-circle me-1"></i>
                            Після налаштування бази даних ви зможете використовувати всі функції модуля.
                        </small>
                    </p>
                </div>
            </div>
        </div>
<?php
        return ob_get_clean();
    }
}

/**
 * Регистрация страницы админки для темы (зарезервировано для будущего использования)
 *
 * Упрощенный способ регистрации страницы админки из темы.
 * Сейчас регистрация происходит автоматически через массив $adminThemePages в Theme.php.
 *
 * Разработчикам тем достаточно добавить элемент в массив $adminThemePages в классе темы.
 *
 * @param string $pageId Идентификатор страницы (используется в URL)
 * @param array{menu: array{text: string, icon: string, title?: string}, callback: string, file: string, enabled?: callable|bool|string} $config Конфигурация страницы
 * @return void
 */
if (!function_exists('register_theme_admin_page')) {
    function register_theme_admin_page(string $pageId, array $config): void
    {
        // Эта функция зарезервирована для будущего использования
        // Сейчас регистрация происходит автоматически через массив $adminThemePages в Theme.php
        // Разработчикам тем достаточно добавить элемент в массив $adminThemePages
    }
}
