<?php
/**
 * Компонент статистичних карток
 *
 * Універсальний компонент для відображення статистичних карток у плагінах та системних сторінках.
 * Підтримує різну кількість карток з автоматичним розподілом по колонках.
 *
 * @param array $cards Масив карток, кожна картка містить:
 *   - 'title' => 'Заголовок картки'
 *   - 'value' => 'Значення (може бути HTML)'
 *   - 'icon' => 'назва іконки Font Awesome (без fas fa-)'
 *   - 'color' => 'primary|info|success|warning|danger (за замовчуванням: primary)'
 *   - 'valueClass' => 'h4|h5|h6 (розмір значення, за замовчуванням: h4)'
 * @param int $columnsPerRow Кількість колонок в рядку (за замовчуванням: автоматично на основі кількості карток)
 * @param array $classes Додаткові CSS класи для контейнера
 *
 * Приклад використання:
 * ```php
 * $componentsPath = __DIR__ . '/../../engine/interface/admin-ui/components/';
 * $cards = [
 *     [
 *         'title' => 'Всього файлів',
 *         'value' => '14',
 *         'icon' => 'file',
 *         'color' => 'primary'
 *     ],
 *     [
 *         'title' => 'Загальний розмір',
 *         'value' => '10,19 KB',
 *         'icon' => 'hdd',
 *         'color' => 'info'
 *     ],
 *     [
 *         'title' => 'Статус',
 *         'value' => '<span class="text-success">Активний</span>',
 *         'icon' => 'check-circle',
 *         'color' => 'success',
 *         'valueClass' => 'h5'
 *     ]
 * ];
 * include $componentsPath . 'stats-cards.php';
 * ```
 */
if (! isset($cards) || ! is_array($cards)) {
    $cards = [];
}
if (! isset($classes)) {
    $classes = [];
}

// Визначаємо кількість колонок та розподіл
$cardCount = count($cards);

// Если columnsPerRow не установлен извне, используем автоматическое определение
if (!isset($columnsPerRow) || $columnsPerRow === null || $columnsPerRow === '') {
    // Автоматичне визначення кількості колонок
    if ($cardCount <= 2) {
        // Для 1-2 карток: розподіл порівну
        $columnsPerRow = $cardCount;
        $fixedWidth = false;
    } elseif ($cardCount <= 5) {
        // Для 3-5 карток: розтягнути на всю ширину
        $columnsPerRow = $cardCount;
        $fixedWidth = true; // Фіксована ширина для рівномірного розподілу
    } else {
        // Для 6+ карток: 4 в рядку, решта переноситься
        $columnsPerRow = 4;
        $fixedWidth = true;
    }
} else {
    // Если columnsPerRow установлен извне, используем его
    if (!isset($fixedWidth)) {
        $fixedWidth = true; // По умолчанию используем fixedWidth для переданного columnsPerRow
    }
}

// Розраховуємо розмір колонки
if ($fixedWidth) {
    // Використовуємо фіксовану ширину з flex для рівномірного розподілу
    $columnClass = 'col-12 col-md';
    // Если columnsPerRow установлен явно, используем его для расчета minWidth
    if (isset($columnsPerRow) && $columnsPerRow > 0 && $columnsPerRow <= $cardCount) {
        $minWidth = 100 / $columnsPerRow;
    } else {
        $minWidth = $cardCount <= 5 ? (100 / $cardCount) : 25; // Для 5- карток - розподіл порівну, для 6+ - 25% (4 в рядку)
    }
} else {
    // Для 1-2 карток використовуємо стандартну Bootstrap сітку
    $columnSize = 12 / $columnsPerRow;
    $columnClass = 'col-12 col-md-' . $columnSize;
    $minWidth = null;
}

// Колір лівої смуги за замовчуванням
$defaultColors = [
    'primary' => '#0073aa',
    'info' => '#00a0d2',
    'success' => '#46b450',
    'warning' => '#ffb900',
    'danger' => '#dc3232'
];

$containerClasses = ['stats-cards'];
if (! empty($classes)) {
    $containerClasses = array_merge($containerClasses, $classes);
}
$containerClass = implode(' ', array_map('htmlspecialchars', $containerClasses));

// Додаємо атрибути для CSS
$dataAttributes = '';
if ($fixedWidth && isset($minWidth)) {
    $dataAttributes = ' data-fixed-width="true" data-min-width="' . htmlspecialchars($minWidth) . '%"';
}
if ($cardCount > 5) {
    $dataAttributes .= ' data-cards-per-row="' . htmlspecialchars($columnsPerRow) . '"';
}
?>
<div class="<?= $containerClass ?>" data-cards-count="<?= $cardCount ?>"<?= $dataAttributes ?>>
    <div class="row g-3">
        <?php foreach ($cards as $card): ?>
            <?php
            $title = $card['title'] ?? '';
            $value = $card['value'] ?? '';
            $icon = $card['icon'] ?? '';
            $color = $card['color'] ?? 'primary';
            $valueClass = $card['valueClass'] ?? 'h4';

            // Колір лівої смуги
            $borderColor = $defaultColors[$color] ?? $defaultColors['primary'];
            $borderClass = 'border-left-' . $color;
            ?>
            <div class="<?= $columnClass ?>">
                <div class="card <?= $borderClass ?> h-100">
                    <div class="card-body">
                        <div class="row align-items-center">
                            <div class="col">
                                <?php if (! empty($title)): ?>
                                <div class="stats-card-title">
                                    <?= htmlspecialchars($title) ?>
                                </div>
                                <?php endif; ?>

                                <?php if (isset($value) && $value !== '' && $value !== null): ?>
                                <div class="stats-card-value <?= htmlspecialchars($valueClass) ?> mb-0 font-weight-bold">
                                    <?= $value // Може містити HTML ?>
                                </div>
                                <?php endif; ?>
                            </div>

                            <?php if (! empty($icon)): ?>
                            <div class="col-auto">
                                <i class="fas fa-<?= htmlspecialchars($icon) ?> stats-card-icon"></i>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    </div>
</div>

<!-- Стилі винесені в SCSS: components/_stats-cards.scss -->

<?php
// Додаємо CSS змінні для фіксованої ширини
if ($fixedWidth && isset($minWidth)): ?>
<script>
(function() {
    const container = document.querySelector('.stats-cards[data-fixed-width="true"]:last-of-type');
    if (container) {
        const root = document.documentElement;
        root.style.setProperty('--card-min-width', '<?= $minWidth ?>%');
    }
})();
</script>
<?php endif; ?>
