<?php
/**
 * Компонент інформаційного блоку
 *
 * Універсальний компонент для відображення інформаційних блоків у плагінах та системних сторінках.
 *
 * @param string $title Заголовок блоку
 * @param string $titleIcon Іконка для заголовка (наприклад: 'info-circle' або повний клас 'fas fa-info-circle')
 * @param array $sections Масив секцій, кожна секція містить:
 *   - 'title' => 'Заголовок секції'
 *   - 'icon' => 'назва іконки Font Awesome (без fas fa-)'
 *   - 'iconColor' => 'primary|info|success|warning|danger (за замовчуванням: primary)'
 *   - 'items' => ['Текст пункту 1', 'Текст пункту 2', ...]
 *   - 'checkIcon' => true/false (показувати іконку галочки, за замовчуванням: true)
 * @param array $classes Додаткові CSS класи для контейнера
 *
 * Приклад використання:
 * ```php
 * $componentsPath = __DIR__ . '/../../engine/interface/admin-ui/components/';
 * $title = 'Про кеш системи';
 * $sections = [
 *     [
 *         'title' => 'Що таке кеш:',
 *         'icon' => 'question-circle',
 *         'iconColor' => 'primary',
 *         'items' => [
 *             'Зберігає результати обчислень та запитів до БД',
 *             'Прискорює роботу системи',
 *             'Автоматично оновлюється при зміні даних'
 *         ]
 *     ],
 *     [
 *         'title' => 'Коли очищати:',
 *         'icon' => 'clock',
 *         'iconColor' => 'info',
 *         'items' => [
 *             'Після оновлення системи',
 *             'При проблемах з відображенням'
 *         ]
 *     ]
 * ];
 * include $componentsPath . 'info-block.php';
 * ```
 */
if (! isset($title)) {
    $title = 'Інформація';
}
if (! isset($titleIcon)) {
    $titleIcon = '';
}
if (! isset($sections) || ! is_array($sections)) {
    $sections = [];
}
if (! isset($classes)) {
    $classes = [];
}

$containerClasses = ['info-block'];
if (! empty($classes)) {
    $containerClasses = array_merge($containerClasses, $classes);
}
$containerClass = implode(' ', array_map('htmlspecialchars', $containerClasses));

// Колір іконок за замовчуванням
$defaultIconColors = [
    'primary' => '#0073aa',
    'info' => '#00a0d2',
    'success' => '#46b450',
    'warning' => '#ffb900',
    'danger' => '#dc3232'
];
?>
<div class="<?= $containerClass ?>">
    <div class="card border-0">
        <?php if (! empty($title)): ?>
        <div class="card-header border-bottom">
            <h5 class="mb-0">
                <?php if (! empty($titleIcon)): ?>
                    <?php
                    // Якщо іконка не містить 'fa-', додаємо стандартний префікс
                    if (strpos($titleIcon, 'fa-') === false && strpos($titleIcon, ' ') === false) {
                        $titleIconClass = 'fas fa-' . htmlspecialchars($titleIcon);
                    } else {
                        $titleIconClass = htmlspecialchars($titleIcon);
                    }
                    ?>
                    <i class="<?= $titleIconClass ?> me-2"></i>
                <?php endif; ?>
                <?= htmlspecialchars($title) ?>
            </h5>
        </div>
        <?php endif; ?>

        <div class="card-body">
            <?php if (! empty($sections)): ?>
                <div class="row g-3">
                    <?php foreach ($sections as $section): ?>
                        <?php
                        $sectionTitle = $section['title'] ?? '';
                        $sectionIcon = $section['icon'] ?? '';
                        $iconColor = $section['iconColor'] ?? 'primary';
                        $items = $section['items'] ?? [];
                        $checkIcon = isset($section['checkIcon']) ? (bool)$section['checkIcon'] : true;

                        // Колір іконки
                        $iconColorValue = $defaultIconColors[$iconColor] ?? $defaultIconColors['primary'];
                        ?>
                        <div class="col-12 col-md-6">
                            <?php if (! empty($sectionTitle)): ?>
                            <h6 class="info-block-section-title mb-2" style="color: <?= $iconColorValue ?>;">
                                <?php if (! empty($sectionIcon)): ?>
                                    <i class="fas fa-<?= htmlspecialchars($sectionIcon) ?> me-2"></i>
                                <?php endif; ?>
                                <?= htmlspecialchars($sectionTitle) ?>
                            </h6>
                            <?php endif; ?>

                            <?php if (! empty($items)): ?>
                            <ul class="info-block-list">
                                <?php foreach ($items as $index => $item): ?>
                                    <li class="info-block-item">
                                        <?php if ($checkIcon): ?>
                                            <i class="fas fa-check text-success me-2"></i>
                                        <?php endif; ?>
                                        <?= htmlspecialchars($item) ?>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Стилі винесені в SCSS: components/_info-block.scss -->
