(function() {
    'use strict';
    const style = document.createElement('style');
    style.textContent = `
        .falling-leaf {
            position: fixed;
            pointer-events: none;
            z-index: 9999;
            border-radius: 50% 0 50% 50%;
            opacity: 0.8;
            transform-origin: center;
        }
        .leaf-color-1 { background: linear-gradient(45deg, #FF6B35, #FF8E53); }
        .leaf-color-2 { background: linear-gradient(45deg, #FFD166, #FFB347); }
        .leaf-color-3 { background: linear-gradient(45deg, #EF476F, #FF6B6B); }
        .leaf-color-4 { background: linear-gradient(45deg, #06D6A0, #4ECDC4); }
        .leaf-color-5 { background: linear-gradient(45deg, #118AB2, #06D6A0); }
    `;
    document.head.appendChild(style);
    
    class FallingLeaves {
        constructor() {
            this.leaves = [];
            this.leafCount = 0;
            this.maxLeaves = 25;
            this.init();
        }
        
        init() {
            setInterval(() => {
                if (this.leafCount < this.maxLeaves) {
                    this.createLeaf();
                }
            }, 600);
            
            this.animate();
        }
        
        createLeaf() {
            const leaf = document.createElement('div');
            leaf.className = 'falling-leaf';
            
            const size = Math.random() * 25 + 15;
            
            const colorClass = `leaf-color-${Math.floor(Math.random() * 5) + 1}`;
            leaf.classList.add(colorClass);
            
            const startX = Math.random() * window.innerWidth;
            const fallSpeed = Math.random() * 2 + 1;
            const rotationSpeed = (Math.random() - 0.5) * 3;
            const swayAmplitude = Math.random() * 80 + 40;
            const swayFrequency = Math.random() * 0.015 + 0.01;
            
            leaf.style.width = `${size}px`;
            leaf.style.height = `${size * 0.8}px`;
            leaf.style.left = `${startX}px`;
            leaf.style.top = `-50px`;        
            document.body.appendChild(leaf);       
            this.leaves.push({
                element: leaf,
                x: startX,
                y: -50,
                speed: fallSpeed,
                rotation: Math.random() * 360,
                rotationSpeed: rotationSpeed,
                swayAmplitude: swayAmplitude,
                swayFrequency: swayFrequency,
                time: Math.random() * 100
            });
            
            this.leafCount++;
        }       
        animate() {
            for (let i = this.leaves.length - 1; i >= 0; i--) {
                const leaf = this.leaves[i];
                leaf.time += 0.05;
                leaf.y += leaf.speed;
                leaf.x += Math.sin(leaf.time * leaf.swayFrequency) * 0.3;
                leaf.rotation += leaf.rotationSpeed;
                leaf.element.style.left = `${leaf.x}px`;
                leaf.element.style.top = `${leaf.y}px`;
                leaf.element.style.transform = `rotate(${leaf.rotation}deg)`;
                if (leaf.y > window.innerHeight - 100) {
                    leaf.element.style.opacity = `${(window.innerHeight - leaf.y) / 100}`;
                }
                if (leaf.y > window.innerHeight + 50) {
                    leaf.element.remove();
                    this.leaves.splice(i, 1);
                    this.leafCount--;
                }
            }           
            requestAnimationFrame(() => this.animate());
        }
    }
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            new FallingLeaves();
        });
    } else {
        new FallingLeaves();
    }  
})();