<?php

namespace TC\PaygateMorune\Payment;

use XF;
use XF\Entity\PaymentProfile;
use XF\Entity\PurchaseRequest;
use XF\Http\Request;
use XF\Mvc\Controller;
use XF\Payment\AbstractProvider;
use XF\Payment\CallbackState;
use XF\Purchasable\Purchase;

class Morune extends AbstractProvider
{
	/**
	 * @return string
	 */
	public function getTitle()
	{
		return 'Morune';
	}

	/**
	 * @return string
	 */
	public function getApiEndpoint()
	{
		return 'https://api.morune.com';
	}

	/**
	 * @param array $options
	 * @param array $errors
	 *
	 * @return bool
	 */
	public function verifyConfig(array &$options, &$errors = [])
	{
		if (empty($options['shop_id']) || empty($options['secret_key']) || empty($options['secret_key_2']))
		{
			$errors[] = XF::phrase('tc_morune_you_must_provide_all_data');
		}
		if ($errors)
		{
			return false;
		}

		return true;
	}

	/**
	 * @param PurchaseRequest $purchaseRequest
	 * @param Purchase        $purchase
	 *
	 * @return array
	 */
	protected function getPaymentParams(PurchaseRequest $purchaseRequest, Purchase $purchase)
	{
		$paymentProfile = $purchase->paymentProfile;

		return [
			'amount'   => number_format($purchaseRequest->cost_amount, 2, '.', ''),
			'order_id' => $purchaseRequest->request_key,
			'currency' => $purchaseRequest->cost_currency,
			'shop_id'  => $paymentProfile->options['shop_id'],

			'hook_url'    => $this->getCallbackUrl(),
			'success_url' => $purchase->returnUrl,
			'fail_url'    => $purchase->cancelUrl
		];
	}

	/**
	 * @param Controller      $controller
	 * @param PurchaseRequest $purchaseRequest
	 * @param Purchase        $purchase
	 *
	 * @return XF\Mvc\Reply\AbstractReply
	 */
	public function initiatePayment(Controller $controller, PurchaseRequest $purchaseRequest, Purchase $purchase)
	{
		$profileOptions = $purchase->paymentProfile->options;

		// API request to create payment
		$params = $this->getPaymentParams($purchaseRequest, $purchase);

		$response = XF::app()->http()->client()->post($this->getApiEndpoint() . '/invoice/create', [
			'headers'    => [
				'Content-Type' => 'application/json',
				'Accept'       => 'application/json',
				'x-api-key'    => $profileOptions['secret_key']
			],
			'json'       => $params,
			'exceptions' => false
		]);

		if ($response)
		{
			$responseData = json_decode($response->getBody()->getContents(), true);

			if ($response->getStatusCode() == 200)
			{
				$invoice = $responseData['data'];

				// Save payment UUID and redirect user
				$purchaseRequest->fastUpdate('provider_metadata', $invoice['id']);

				return $controller->redirect($invoice['url']);
			}
			else
			{
				return $controller->error($responseData['error'] ?? XF::phrase('something_went_wrong_please_try_again'));
			}
		}

		return $controller->error(XF::phrase('something_went_wrong_please_try_again'));
	}

	/**
	 * @param Request $request
	 *
	 * @return CallbackState
	 */
	public function setupCallback(Request $request)
	{
		$state = new CallbackState();

		$state->providerId = $request->filter('_xfProvider', 'str');
		$state->ip = $request->getIp();
		$state->rawInput = $request->getInputRaw();

		$state->inputArray = json_decode($state->rawInput, true) ?? [];
		$state->inputFiltered = $request->getInputFilterer()->filterArray($state->inputArray, [
			'invoice_id'    => 'str',
			'status'        => 'str',
			'amount'        => 'str',
			'currency'      => 'str',
			'order_id'      => 'str',
			'pay_service'   => 'str',
			'payer_details' => 'str',
			'custom_fields' => 'json-array',
			'type'          => 'int',
			'credited'      => 'str',
			'pay_time'      => 'str',
			'code'          => 'int',
			'reject_time'   => 'str',
			'refund_amount' => 'str',
			'refund_reason' => 'str',
			'refund_time'   => 'str'
		]);

		$state->transactionId = $state->inputFiltered['invoice_id'] ?? null;
		$state->requestKey = $state->inputFiltered['order_id'] ?? null;

		$state->signature = $request->getServer('HTTP_X_API_SHA256_SIGNATURE', '');

		$state->costAmount = $state->inputFiltered['amount'] ?? null;
		$state->costCurrency = $state->inputFiltered['currency'] ?? null;

		$state->type = $state->inputFiltered['type'] ?? null;
		$state->code = $state->inputFiltered['code'] ?? null;

        $state->status = $state->inputFiltered['status'] ?? 'unknown';

		return $state;
	}

	/**
	 * @param CallbackState $state
	 *
	 * @return bool
	 */
	public function validateCallback(CallbackState $state): bool
	{
		// Additional check because XF doesn't do it. Why, Chris?
		if ($state->transactionId && $state->requestKey && $state->getPaymentProfile())
		{
			if ($state->getPaymentProfile()->provider_id == $state->providerId)
			{
                if ($state->status != 'success') {
                    $state->logType = 'info';
                    $state->logMessage = "Unsuccessful status for order $state->requestKey - $state->status";

                    return false;
                }
				return parent::validateCallback($state);
			}

			$state->logType = 'info';
			$state->logMessage = 'Invalid provider.';

			return false;
		}

		$state->logType = 'info';
		$state->logMessage = 'No Transaction ID or Request Key. No action to take.';

		return false;
	}

	/**
	 * @param CallbackState $state
	 *
	 * @return bool
	 */
	public function validatePurchasableData(CallbackState $state): bool
	{
		$paymentProfile = $state->getPaymentProfile();
		$profileOptions = $paymentProfile->options;

		$options = $paymentProfile->options;
		if (!empty($options['secret_key_2']))
		{
			if (!empty($state->signature))
			{
				/** @var array $inputArray */
				$inputArray = $state->inputArray;
				ksort($inputArray);

				$generatedSignature = hash_hmac('sha256', json_encode($inputArray), $profileOptions['secret_key_2']);
				if (hash_equals($state->signature, $generatedSignature))
				{
					return true;
				}

				$state->logType = 'error';
				$state->logMessage = 'Invalid signature.';

				return false;
			}

			$state->logType = 'error';
			$state->logMessage = 'Empty signature.';

			return false;
		}

		$state->logType = 'error';
		$state->logMessage = 'Invalid additional_key in forum settings.';

		return false;
	}

	/**
	 * @param CallbackState $state
	 *
	 * @return bool
	 */
	public function validateCost(CallbackState $state)
	{
		$purchaseRequest = $state->getPurchaseRequest();

		$costValidated = round($state->costAmount, 2) == round($purchaseRequest->cost_amount, 2);
		if ($costValidated)
		{
			return true;
		}

		$state->logType = 'error';
		$state->logMessage = 'Invalid Cost or Currency.';

		return false;
	}

	/**
	 * @param CallbackState $state
	 */
	public function getPaymentResult(CallbackState $state)
	{
		if ($state->code == 1 && $state->type == 1)
		{
			$state->paymentResult = CallbackState::PAYMENT_RECEIVED;
		}
	}

	/**
	 * @param CallbackState $state
	 */
	public function prepareLogData(CallbackState $state)
	{
		$state->logDetails = [
			'ip'            => $state->ip,
			'request_time'  => XF::$time,
			'inputFiltered' => $state->inputFiltered,
			'inputJson'     => $state->rawInput
		];
	}

	/**
	 * @var array
	 */
	protected $supportedCurrencies = [
		'RUB', 'USD', 'EUR', 'UAH'
	];

	/**
	 * @param PaymentProfile $paymentProfile
	 * @param                $unit
	 * @param                $amount
	 * @param int            $result
	 *
	 * @return bool
	 */
	public function supportsRecurring(PaymentProfile $paymentProfile, $unit, $amount, &$result = self::ERR_NO_RECURRING)
	{
		$result = self::ERR_NO_RECURRING;

		return false;
	}

	/**
	 * @return array
	 */
	protected function getSupportedRecurrenceRanges()
	{
		return [];
	}

	/**
	 * @param PaymentProfile $paymentProfile
	 * @param                $currencyCode
	 *
	 * @return bool
	 */
	public function verifyCurrency(PaymentProfile $paymentProfile, $currencyCode): bool
	{
		$addOns = XF::app()->container('addon.cache');

		if (!empty($paymentProfile->options['tc_cu_exchange_to'])
			&& array_key_exists('TC/CurrencyUtils', $addOns)
			&& XF::registry()->exists('tcCurrencies'))
		{
			return array_key_exists($currencyCode, XF::registry()->get('tcCurrencies'));
		}

		return in_array($currencyCode, $this->supportedCurrencies);
	}
}