/*
 * Blobby
 *
 * A puzzle platformer game.
 *
 * (c) 2014 Danijel Durakovic
 *
 */

Blobby = (function() {
	// iterator
	var Iterate = function(collection, callback) {
		for (var obj in collection)
			if (collection.hasOwnProperty(obj))
				callback(obj, collection[obj]);
	};

	// assets module
	var Assets = (function() {
		// collection of media assets to be used in-game
		// these include graphics and audio assets
		var assets = {
			// graphics
			gfx_ui: 'gfx/ui.png',
			gfx_icons: 'gfx/icons.png',
			gfx_labels: 'gfx/labels.png',
			gfx_num: 'gfx/num.png',
			gfx_master: 'gfx/master.png',
			gfx_objects: 'gfx/objects.png',
			gfx_blobby: 'gfx/blobby.png',
			gfx_fx: 'gfx/fx.png',
			gfx_friends: 'gfx/friends.png',
			gfx_pstory: 'gfx/pstory.png',
			gfx_stageselect: 'gfx/stageselect.png',
			gfx_pinfo: 'gfx/pinfo.png',
			gfx_pwin: 'gfx/pwin.png',
			gfx_pvictory: 'gfx/pvictory.png',
			gfx_hudlabels: 'gfx/hudlabels.png',
			gfx_enemies: 'gfx/enemies.png',
			// tracks
			track_1: 'music/DST-LettuceCandy.ogg',
			track_2: 'music/DST-MoonBeach.ogg',
			track_3: 'music/DST-Probly.ogg'
		};

		// level count - number of level files ot be loaded
		// levels will be loaded in sequential order and
		// separately from media assets to retain level order
		// level format: lvl/level1.json to lvl/levelN.json
		var levelCount = 12;

		// compiled collection of assets
		var loadedAssets = {};

		// compiled array of json levels
		var loadedLevels = [];

		// retreives a filename's extension
		var GetExt = function(filename) {
			return filename.split('.').pop().toLowerCase();
		};

		return {
			// load function
			// usage: Load(finished callback, [progress callback])
			Load: function(finished, progress) {
				// check for ogg support
				var loadOgg = Music.HasOggSupport();

				// track item loading
				var itemsLoaded = 0;
			
				// count items to load
				var itemsToLoad = levelCount;
				Iterate(assets, function(key, property) {
					var ext = GetExt(property);
					if (['png', 'jpg', 'jpeg'].indexOf(ext) > -1 ||
						(loadOgg && 'ogg' === ext))
						itemsToLoad++;
				});

				// advance the progress when individual resource is loaded
				var Advance = function() {
					// increment counter
					itemsLoaded++;
					// track progress and finish when all items load
					if (progress instanceof Function)
						progress(itemsLoaded, itemsToLoad);

					if (itemsLoaded == itemsToLoad)
						finished();
				};

				// load media assets
				Iterate(assets, function(key, property) {
					var ext = GetExt(property);
					if (['png', 'jpg', 'jpeg'].indexOf(ext) > -1) {
						// load image asset
						var loadImg = loadedAssets[key] = new Image();
						loadImg.src = property;
						loadImg.addEventListener('load', Advance, false);
					}
					else if (loadOgg && 'ogg' === ext) {
						// load audio asset
						var loadAudio = loadedAssets[key] = new Audio();
						loadAudio.src = property;
						loadAudio.addEventListener('canplaythrough', Advance, false);
					}
				});

				// load level files
				(function LoadLevel(levelIndex) {
					var levelFile = 'lvl/level' + levelIndex + '.json';
					// request json file
					var xhr = new XMLHttpRequest;
					xhr.open('get', levelFile, true);
					xhr.send(null);
					xhr.onreadystatechange = function() {
						if (xhr.readyState === 4 && xhr.status === 200) {
							// parse json data from level file
							var json = xhr.responseText;
							loadedLevels.push(JSON.parse(json));
							Advance();
							if (levelIndex < levelCount) {
								// load next level in sequence
								LoadLevel(levelIndex + 1);
							}
						}
					};
				})(1); // start loading at first level
			},
			// get media asset
			Get: function(asset) {
				return loadedAssets[asset];
			},
			// get level
			GetLevel: function(levelIndex) {
				if (levelIndex >= 1 && levelIndex <= levelCount)
					return loadedLevels[levelIndex - 1];
			},
			// get level count
			GetLevelCount: function() {
				return levelCount;
			},
			// creates an array of tracks and returns it
			GetTracks: function() {
				var trackList = [];
				Iterate(loadedAssets, function(key, property) {
					if (key.substring(0, 6).toLowerCase() == 'track_')
						trackList.push(property);
				});
				return trackList;
			}
		};
	})();

	// drawing module
	var Draw = (function() {
		// drawing context
		var ctx;

		return {
			// set drawing context
			SetContext: function(context) {
				ctx = context;
			},
			// sets global alpha
			// usage: SetAlpha ([alpha = 1.0])
			SetAlpha: function(alpha) {
				if (typeof alpha == 'undefined')
					alpha = 1.0;
				ctx.globalAlpha = alpha;
			},
			// draws text
			// usage: Text(text, x, y, [fillStyle], [font])
			Text: function(text, x, y, fillStyle, font) {
				if (typeof fillStyle == 'undefined')
					fillStyle = '#fff';
				if (typeof font == 'undefined')
					font = '12px sans-serif';
				ctx.textBaseline = 'top';
				ctx.fillStyle = fillStyle;
				ctx.font = font;
				ctx.fillText(text, x, y);
			},
			// draws a rectangle
			// usage: Rect(x, y, width, height, [fillStyle], [lineWidth])
			Rect: function(x, y, w, h, fillStyle, lineWidth) {
				if (typeof fillStyle == 'undefined')
					fillStyle = '#fff';
				if (typeof lineWidth == 'undefined')
					lineWidth = 1;
				ctx.strokeStyle = fillStyle;
				ctx.lineWidth = lineWidth;
				ctx.strokeRect(x, y, w, h);
			},
			// draws a filled rectangle
			// usage: RectFill(x, y, width, height, [fillStyle])
			RectFill: function(x, y, w, h, fillStyle) {
				if (typeof fillStyle == 'undefined')
					fillStyle = '#fff';
				ctx.fillStyle = fillStyle;
				ctx.fillRect(x, y, w, h);
			},
			// draws a line
			// usage: Line(x1, y1, x2, y2, [strokeStyle], [lineWidth])
			Line: function(x1, y1, x2, y2, strokeStyle, lineWidth) {
				if (typeof strokeStyle == 'undefined')
					strokeStyle = '#fff';
				if (typeof lineWidth == 'undefined')
					lineWidth = 1;
				ctx.strokeStyle = strokeStyle;
				ctx.lineWidth = lineWidth;
				ctx.moveTo(x1, y1);
				ctx.lineTo(x2, y2);
				ctx.stroke();
			},
			// draws an image
			// usage: Image(image, x, y, [width], [height])
			Image: function(image, x, y, w, h) {
				if (typeof w != 'undefined' && typeof h != 'undefined') {
					ctx.drawImage(image, x, y, w, h);
				}
				else {
					ctx.drawImage(image, x, y);
				}
			},
			// draws a tile
			// usage: Image(image, x, y, width, height, sx, sy)
			Tile: function(image, x, y, w, h, sx, sy) {
				ctx.drawImage(image, sx, sy, w, h, x, y, w, h);
			},
			// draws numeric glyphs
			// usage: Num(glyph_tiles, number, x, y, [color index])
			Num: function(glyphs, number, x, y, ci) {
				if (typeof ci == 'undefined')
					ci = 0;
				var numstr = number.toString();
				for (var i = 0; i < numstr.length; i++) {
					var n = parseInt(numstr[i], 10);
					ctx.drawImage(glyphs, 18 * n, ci * 30, 18, 30, x + 18 * i, y, 18, 30);
				}
			}
		};
	})();

	// music module
	var Music = (function() {
		// ogg support flag
		var supportsOgg = false;

		// test browser for ogg support
		(function TestOgg() {
			var audio = document.createElement('audio');
			supportsOgg = (typeof audio.canPlayType == typeof Function &&
				audio.canPlayType('audio/ogg') !== '');
		})();

		// list of tracks
		var trackList;

		// currently playing track
		var track = null;
		
		// get a random track index
		var RandomTrack = function() {
			return Math.floor(Math.random() * trackList.length);
		};

		return {
			// returns if browser supports ogg playback
			HasOggSupport: function() {
				return supportsOgg;
			},
			// set playable tracks
			SetTracks: function(_trackList) {
				if (!supportsOgg) return;
				trackList = _trackList;
			},
			// plays a track
			PlayTrack: function(trackIndex) {
				if (!supportsOgg || trackList.length == 0) return;
				var self = this;
				track = trackList[trackIndex];
				track.play();
				var TrackEnded = function() {
					// play the next track
					trackIndex++;
					if (trackIndex >= trackList.length)
						trackIndex = 0;
					track.pause();
					track.currentTime = 0;
					self.PlayTrack(trackIndex);
				};
				track.removeEventListener('ended', TrackEnded);
				track.addEventListener('ended', TrackEnded);
			},
			// start track playback
			// the track is selected at random
			StartPlayback: function() {
				if (!supportsOgg) return;
				if (track != null) {
					track.pause();
					track.currentTime = 0;
				}
				this.PlayTrack(RandomTrack());
			},
			// stops playback
			StopPlayback: function() {
				if (!supportsOgg) return;
				if (track != null) {
					track.pause();
					track.currentTime = 0;
					track = null;
				}
			}
		};
	})();

	// game module
	var Game = (function() {
		// reference to canvas
		var canvas;

		// graphics
		var gfx_ui, gfx_icons, gfx_labels, gfx_num, gfx_master, gfx_objects,
			gfx_blobby, gfx_fx, gfx_friends, gfx_hudlabels, gfx_enemies;

		// current stage
		var stage = 1;

		// game configuration settings
		var config = {
			unlocked: 1,
			music: true
		};

		// save config to local storage
		var SaveConfig = function() {
			localStorage.setItem('blobby', JSON.stringify(config));
		};

		// load config from local storage
		var LoadConfig = function() {
			var cfg = localStorage.getItem('blobby');
			if (cfg)
				config = JSON.parse(cfg);
		}

		// counter based timer class
		var Timer = function(interval) {
			// tick frequency
			var interval = interval;
			// count
			var count = 0;
			// run ticker
			this.Run = function() {
				count++;
				if (count > interval)
					count = 0;
			};
			// tick function
			this.Tick = function() {
				return (count == 0);
			};
			// reset the timer
			this.Reset = function() {
				count = 0;
			};
		};

		// transitions
		var Transition = {
			ScrollCameraUp: function() {
				var seq = Scene.GetCameraY();

				this.Logic = function() {
					seq -= 18;

					Scene.SetCameraY((seq >= 0) ? seq : 0);

					if (seq <= 0) {
						return true;
					}

					return false;
				};
			},
			MoveEditPanel: function() {
				var seq = 0;

				this.Logic = function() {
					seq++;
					EditPanel.Move(4);
					if (seq >= 36)
						return true;
					return false;
				};
			},
			MoveStopButton: function() {
				var seq = 0;

				this.Logic = function() {
					seq++;
					el.btnStop.GetProps().y+=2;
					el.btnStart.GetProps().y+=2;
					if (seq >= 24)
						return true;
					return false;
				};
			},
			ShowPopup: function() {
				var seq = 0;

				this.Logic = function() {
					seq++;
					var alpha = (seq * 3) / 100;
					Popup.SetBgAlpha(alpha);
					if (alpha >= 0.9) {
						Popup.SetActive(true);
						return true;
					}
					return false;
				};
			},
		};

		// array of active transitions
		var activeTransitions = [];

		// returns if a point is within rectangle
		var PointInRect = function(x, y, rx, ry, rw, rh) {
			return (x >= rx && x <= rx + rw &&
					y >= ry && y <= ry + rh);
		};

		// translates tap coordinates to game coordinates
		var TranslateCoords = function(e, agent) {
			var ratio = Math.max(canvas.width / window.innerWidth, canvas.height / window.innerHeight);
			var bounds = canvas.getBoundingClientRect();

			var px = (agent === 'touch') ? e.changedTouches[0].clientX : e.clientX;
			var py = (agent === 'touch') ? e.changedTouches[0].clientY : e.clientY;

			return {
				x: Math.floor((px - bounds.left) * ratio),
				y: Math.floor((py - bounds.top) * ratio)
			};
		};

		// touch button
		var Button = function(p, c) {
			// user event flag
			var userEvent = false;

			// states
			var active = false;
			var pressed = false;

			// enabled flag
			var enabled = true;

			// press event
			var PressEvent = function(e, agent) {
				if (e.preventDefault) e.preventDefault();
				if (e.stopPropagation) e.stopPropagation();
				if (!enabled || activeTransitions.length > 0
					|| (!c.mainButton && Popup.IsActive())) return false;
				// prevent default actions
				e.preventDefault();
				var coords = TranslateCoords(e, agent);

				if (PointInRect(coords.x, coords.y, p.x, p.y, c.w, c.h)) {
					if (!c.edit) {
						active = true;
						pressed = true;
					}
					else
						userEvent = true;
				}
				return false;
			};

			// move event
			var MoveEvent = function(e, agent) {
				if (e.preventDefault) e.preventDefault();
				if (e.stopPropagation) e.stopPropagation();
				if (!enabled || activeTransitions.length > 0
					|| (!c.mainButton && Popup.IsActive())) return false;
				if (!c.edit && pressed) {
					var coords = TranslateCoords(e, agent);
					active = PointInRect(coords.x, coords.y, p.x, p.y, c.w, c.h);
				}
				return false;
			};

			// release event
			var ReleaseEvent = function(e, agent) {
				if (e.preventDefault) e.preventDefault();
				if (!enabled || activeTransitions.length > 0
					|| (!c.mainButton && Popup.IsActive())) return false;
				var coords = TranslateCoords(e, agent);
				if (!c.edit) {
					if (active && PointInRect(coords.x, coords.y, p.x, p.y, c.w, c.h))
						userEvent = true;
					active = false;
					pressed = false;
				}
				return false;
			};

			// initialize element
			this.Init = function() {
				// add event handlers
				canvas.addEventListener('mousedown',  function(e) { PressEvent(e, 'mouse');   }, false);
				canvas.addEventListener('touchstart', function(e) { PressEvent(e, 'touch');   }, false);
				window.addEventListener('mousemove',  function(e) { MoveEvent(e, 'mouse');    }, false);
				window.addEventListener('touchmove',  function(e) { MoveEvent(e, 'touch');    }, false);
				window.addEventListener('mouseup',    function(e) { ReleaseEvent(e, 'mouse'); }, false);
				window.addEventListener('touchend',   function(e) { ReleaseEvent(e, 'touch'); }, false);
			};

			// draw method
			this.Draw = function() {
				Draw.Tile(gfx_ui, p.x, p.y, c.w, c.h,
					(active) ? c.tileHover.x : c.tileNormal.x,
					(active) ? c.tileHover.y : c.tileNormal.y);
				if (c.icon) {
					Draw.Tile(gfx_icons,
						p.x + (c.w - 48) / 2,
						p.y + (c.h - 48) / 2 - 12,
						48, 48, c.icon.x, c.icon.y);
				}
				if (c.label) {
					Draw.Tile(gfx_labels,
						p.x + (c.w - c.label.w) / 2,
						p.y + 68,
						c.label.w, c.label.h,
						c.label.x, c.label.y);
				}
			};
			
			// user event function
			this.UserEvent = function() {
				if (userEvent) {
					userEvent = false;
					return true;
				}
				return false;
			};

			// get properties
			this.GetProps = function() {
				return p;
			};

			// set active state
			this.SetActive = function(_active) {
				active = _active;
			};

			// set enabled state
			this.SetEnabled = function(_enabled) {
				enabled = _enabled;
			};

		};

		// interface elements
		var el = {
			// menu buttons
			btnStage: new Button({ x: 0, y: 0 }, {
				w: 95, h: 96,
				icon: null,
				tileNormal: { x: 0, y: 96 },
				tileHover: { x: 95, y: 96 },
				label: { x: 0, y: 0, w: 46, h: 18 },
				edit: false,
				mainButton: true
			}),
			btnMusic: new Button({ x: 288, y: 0 }, {
				w: 95, h: 96,
				icon: { x: 240, y: 0 },
				tileNormal: { x: 0, y: 96 },
				tileHover: { x: 95, y: 96 },
				label: { x: 0, y: 18, w: 47, h: 18 },
				edit: false,
				mainButton: true
			}),
			btnInfo: new Button({ x: 385, y: 0 }, {
				w: 95, h: 96,
				icon: { x: 288, y: 0 },
				tileNormal: { x: 0, y: 96 },
				tileHover: { x: 95, y: 96 },
				label: { x: 0, y: 36, w: 32, h: 18 },
				edit: false,
				mainButton: true
			}),
			// stop button
			btnStop: new Button({ x: 0, y: 624 }, {
				w: 480, h: 48,
				tileNormal: { x: 0, y: 288 },
				tileHover: { x: 0, y: 336 }
			}),
			// path editing buttons
			btnGoLeft: new Button({ x: 0, y: 576 }, {
				w: 96, h: 96,
				icon: { x: 0, y: 0 },
				tileNormal: { x: 190, y: 96 },
				tileHover: { x: 286, y: 96 },
				label: { x: 0, y: 54, w: 54, h: 18 },
				edit: true
			}),
			btnGoRight: new Button({ x: 96, y: 576 }, {
				w: 96, h: 96,
				icon: { x: 48, y: 0 },
				tileNormal: { x: 190, y: 96 },
				tileHover: { x: 286, y: 96 },
				label: { x: 0, y: 72, w: 64, h: 18 },
				edit: true
			}),
			btnJump: new Button({ x: 192, y: 576 }, {
				w: 96, h: 96,
				icon: { x: 96, y: 0 },
				tileNormal: { x: 190, y: 96 },
				tileHover: { x: 286, y: 96 },
				label: { x: 0, y: 90, w: 42, h: 18 },
				edit: true
			}),
			btnWait: new Button({ x: 288, y: 576 }, {
				w: 96, h: 96,
				icon: { x: 144, y: 0 },
				tileNormal: { x: 190, y: 96 },
				tileHover: { x: 286, y: 96 },
				label: { x: 0, y: 108, w: 35, h: 18 },
				edit: true
			}),
			btnErase: new Button({ x: 384, y: 576 }, {
				w: 96, h: 96,
				icon: { x: 192, y: 0 },
				tileNormal: { x: 190, y: 96 },
				tileHover: { x: 286, y: 96 },
				label: { x: 0, y: 126, w: 41, h: 18 },
				edit: true
			}),
			// start button
			btnStart: new Button({ x: 0, y: 672 }, {
				w: 480, h: 48,
				tileNormal: { x: 0, y: 192 },
				tileHover: { x: 0, y: 240 }
			}),
		};

		// edit state flag
		var editState = -1;

		// heads up display
		var HUD = {
			// friend to display
			friend: -1,
			// display editing or playing label
			game_mode: 0,
			// draw function
			Draw: function() {
				// draw blobby's friend
				if (this.friend > -1) {
					Draw.Tile(gfx_friends, 122, 24, 48, 48, 0, 96 * this.friend);
				}
				if (this.game_mode === 0)
					Draw.Tile(gfx_hudlabels, 188, 38, 77, 23, 0, 0);
				else if (this.game_mode === 1)
					Draw.Tile(gfx_hudlabels, 188, 38, 77, 23, 0, 23);
			}
		};

		// popup module
		// handles popup screen
		var Popup = (function() {
			// graphics
			var gfx_pstory, gfx_stageselect, gfx_pinfo, gfx_pwin, gfx_pvictory;

			// alpha values
			var bgAlpha = 0.0;

			// state flag
			var state = -1;

			// active flag
			var active = false;

			// deactivates popup
			var Deactivate = function() {
				state = -1;
				active = false;
			};

			// states
			var States = [
				// story
				{
					Draw: function() {
						Draw.Image(gfx_pstory, 0, 96);
					},
					Logic: function() {
					},
					Press: function(cx, cy) {
						if (PointInRect(cx, cy, 0, 96, 480, 624))
							Deactivate();
					}
				},
				// stage selection
				{
					selected: 0,
					hsel: 0,
					pressed: false,
					pressedx: 0,
					pressedy: 0,
					Draw: function() {
						Draw.Tile(gfx_stageselect, 48, 144, 165, 32, 0, 0);
						for (var y = 0; y < 7; y++) {
							for (var x = 0; x < 6; x++) {
								var i = 1 + x + y * 6;
								var posx = 49 + x * 52 + x * 14;
								var posy = 208 + y * 52 + y * 14;
								if (i <= config.unlocked) {
									Draw.Tile(gfx_stageselect, posx, posy, 52, 52, (i === this.hsel) ? 52 : 0, 32);
									Draw.Num(gfx_num, i, ((i < 10)? 17 : 7) + posx, 10 + posy, 1);
									if (i === stage) {
										Draw.Rect(posx-3, posy-3, 58, 58, '#fff');
									}
								}
								else if (i <= Assets.GetLevelCount()) {
									Draw.Tile(gfx_stageselect, posx, posy, 52, 52, 104, 32);
								}
							}
						}
					},
					Logic: function() {
					},
					Press: function(cx, cy) {
						for (var y = 0; y < 7; y++) {
							for (var x = 0; x < 6; x++) {
								var i = 1 + x + y * 6;
								if (i > config.unlocked)
									return;
								var posx = 49 + x * 52 + x * 14;
								var posy = 208 + y * 52 + y * 14;
								if (PointInRect(cx, cy, posx, posy, 52, 52)) {
									this.hsel = this.selected = i;
									this.pressed = true;
									this.pressedx = posx;
									this.pressedy = posy;
									return;
								}
							}
						}
					},
					Move: function(cx, cy) {
						if (this.pressed && this.selected > 0) {
							if (PointInRect(cx, cy, this.pressedx, this.pressedy, 52, 52)) {
								this.hsel = this.selected;
							}
							else {
								this.hsel = 0;
							}
						}
					},
					Release: function(cx, cy) {
						if (this.pressed) {
							if (this.hsel > 0) {
								if (this.selected !== stage) {
									// stop game
									Scene.Stop();
									// load next stage
									stage = this.selected;
									Scene.LoadLevel(Assets.GetLevel(stage));
									// reset scene
									//Scene.Reset();
									if (Scene.GetPlayMode()) {
										// enter editing mode
										EnterEditMode();
										// reset controls
										SetupEditMode();
									}
								}
								// deactivate popup
								Deactivate();
							}
							this.hsel = this.selected = 0;
							this.pressed = false;
						}
					}
				},
				// info screen
				{
					Draw: function() {
						Draw.Image(gfx_pinfo, 0, 96);
					},
					Logic: function() {
					},
					Press: function(cx, cy) {
						if (PointInRect(cx, cy, 0, 96, 480, 624))
							Deactivate();
					}
				},
				// victory screen
				{
					Draw: function() {
						Draw.Image(gfx_pwin, 0, 96);
						Draw.Num(gfx_num, stage + 1, 310, 350, 2);
					},
					Logic: function() {
					},
					Press: function(cx, cy) {
						if (PointInRect(cx, cy, 0, 96, 480, 624)) {
							// load next stage
							stage++;
							if (config.unlocked < stage)
								config.unlocked = stage;
							Scene.LoadLevel(Assets.GetLevel(stage));
							// stop game
							Scene.Stop();
							// initiate edit mode
							Scene.Reset();
							EnterEditMode();
							SetupEditMode();
							// deactivate popup
							Deactivate();
							// save config
							SaveConfig();
						}
					}
				},
				// game complete screen
				{
					Draw: function() {
						Draw.Image(gfx_pvictory, 0, 96);
					},
					Logic: function() {
					},
					Press: function(cx, cy) {
						if (PointInRect(cx, cy, 0, 96, 480, 624)) {
							// return to edit mode
							Scene.Stop();
							Scene.Reset();
							EnterEditMode();
							SetupEditMode();
							// deactivate popup
							Deactivate();
						}
					}
				}
			];

			var PressEvent = function(e, agent) {
				if (e.preventDefault) e.preventDefault();
				if (e.stopPropagation) e.stopPropagation();
				if (activeTransitions.length > 0) return false;
				var coords = TranslateCoords(e, agent);
				if (active && state >= 0 && state <= 4 &&
					States[state].Press instanceof Function) {
					States[state].Press(coords.x, coords.y);
				}
				return false;
			};

			var MoveEvent = function(e, agent) {
				if (e.preventDefault) e.preventDefault();
				if (e.stopPropagation) e.stopPropagation();
				if (activeTransitions.length > 0) return false;
				var coords = TranslateCoords(e, agent);
				if (active && state >= 0 && state <= 4 &&
					States[state].Move instanceof Function) {
					States[state].Move(coords.x, coords.y);
				}
				return false;
			};

			var ReleaseEvent = function(e, agent) {
				if (e.preventDefault) e.preventDefault();
				if (e.stopPropagation) e.stopPropagation();
				if (activeTransitions.length > 0) return false;
				var coords = TranslateCoords(e, agent);
				if (active && state >= 0 && state <= 4 &&
					States[state].Release instanceof Function) {
					States[state].Release(coords.x, coords.y);
				}
				return false;
			};

			return {
				Init: function() {
					gfx_pstory = Assets.Get('gfx_pstory');
					gfx_stageselect = Assets.Get('gfx_stageselect');
					gfx_pinfo = Assets.Get('gfx_pinfo');
					gfx_pwin = Assets.Get('gfx_pwin');
					gfx_pvictory = Assets.Get('gfx_pvictory');
					// register events
					canvas.addEventListener('mousedown',  function(e) { PressEvent(e, 'mouse');   }, false);
					canvas.addEventListener('touchstart', function(e) { PressEvent(e, 'touch');   }, false);
					window.addEventListener('mousemove',  function(e) { MoveEvent(e, 'mouse');    }, false);
					window.addEventListener('touchmove',  function(e) { MoveEvent(e, 'touch');    }, false);
					window.addEventListener('mouseup',    function(e) { ReleaseEvent(e, 'mouse'); }, false);
					window.addEventListener('touchend',   function(e) { ReleaseEvent(e, 'touch'); }, false);
				},
				Draw: function() {
					if (state > -1) {
						if (bgAlpha > 0.0) {
							Draw.SetAlpha(bgAlpha);
							Draw.RectFill(0, 96, 480, 624, '#000');
							Draw.SetAlpha();
						}
						if (active) {
							States[state].Draw();
						}
					}
				},
				Logic: function() {
					if (state > -1) {
						States[state].Logic();
					}
				},
				IsActive: function() {
					return active;
				},
				SetActive: function(_active) {
					active = _active;
				},
				SetState: function(stateIndex) {
					if (stateIndex >= -1 && stateIndex <= 4)
						state = stateIndex;
				},
				GetState: function() {
					return state;
				},
				SetBgAlpha: function(value) {
					bgAlpha = value;
				}
			};
		})();

		// game scene
		var Scene = (function() {
			// camera
			var camera_y = 0;

			// timers
			var tmr = {
				slowanim: new Timer(8),
				anim: new Timer(6),
				walk: new Timer(4),
				gravity: new Timer(2),
			};

			// scene y position
			var scene_y = 96;

			// scene height
			var scene_h = 480;

			// scene dragging flag
			var dragging = false;

			// drag origin
			var origin_y = 0;

			// camera origin
			var origin_c = 0;

			// playing mode
			var playMode = false;

			// tile data
			var TileData = function(background, wall) {
				this.background = background;
				this.wall = wall;
				this.command = -1;
			};

			// grid class
			var Grid = function(h) {
				var w = 10;
				var h = h;

				var data = new Array(w * h);

				// get and set functions
				this.Get = function(x, y) {
					return data[y * w + x];
				};

				this.Set = function(x, y, val) {
					data[y * w + x] = val;
				};

				this.GetW = function() {
					return w;
				};

				this.GetH = function() {
					return h;
				};
			};

			// game grid
			var grid = null;

			// command points
			var cmdPoints = [];

			// animated effect class
			var FX = function(x, y, tiles) {
				// active flag
				this.active = true;

				// animation sequence
				var animSeq = 0;

				// timer
				var fxTmr = new Timer(8);

				// draw the effect
				this.Draw = function() {
					if (this.active) {
						// draw effect tile
						var tile = tiles[animSeq];
						var tilex = x;
						var tiley = y - camera_y;
						// check if tile is within drawable surface
						if (tilex >= -48 && tiley <= scene_h)
							Draw.Tile(gfx_fx, tilex, tiley + scene_y, 48, 48, tile[0] * 48, tile[1] * 48);
					}
				};

				// effect logic
				this.Logic = function() {
					if (this.active) {
						fxTmr.Run();
						if (fxTmr.Tick()) {
							animSeq++;
							if (animSeq > tiles.length - 1)
								this.active = false;
						}
					}
				};
			};

			// array of active effects
			var effects = [];

			// direction constants
			var UP    = 0;
			var DOWN  = 1;
			var LEFT  = 2;
			var RIGHT = 3;

			// retreive direction from string
			var DirectionFromString = function(direction) {
				if (typeof direction != 'undefined') {
					switch(direction.toLowerCase()) {
						case 'up': return UP;
						case 'down': return DOWN;
						case 'left': return LEFT;
						case 'right': return RIGHT;
					}
				}
				return RIGHT;
			};

			// collision constants
			var COLLISION_NONE    = 0;
			var COLLISION_EDGE    = 1;
			var COLLISION_WALL    = 2;
			var COLLISION_COMMAND = 3;

			// axis-aligned bounding box
			var AABB = function(x1, y1, x2, y2) {
				this.x1 = x1;
				this.y1 = y1;
				this.x2 = x2;
				this.y2 = y2;
				// returns if two AABBs intersect
				this.Intersects = function(other) {
					return (this.x2 > other.x1 && this.y2 > other.y1 &&
							this.x1 < other.x2 && this.y1 < other.y2);
				};
			};

			// object-specific bounding box
			var BBOX = function(x, y, w, h) {
				this.x = x;
				this.y = y;
				this.w = w;
				this.h = h;
			};

			// player command constants
			var CMD_NONE = -1;
			var CMD_GOLEFT = 0;
			var CMD_GORIGHT = 1;
			var CMD_JUMP = 2;
			var CMD_WAIT = 3;

			// player
			var Player = {
				// properties
				x: 0,
				y: 0,
				origX: 0, // original position used for reset
				origY: 0,
				facing: RIGHT,
				origFacing: RIGHT,
				animSeq: 0,
				fallvel: 0,
				jumpvel: 0,
				command: CMD_NONE,
				cmdDelay: 0,
				waitSeq: 0,
				// bounding box
				bbox: new BBOX(8, 0, 34, 48),
				// states
				walking: true,
				falling: false,
				jumping: false,
				// killed state
				killed: false,
				// winning state
				winning: false,
				// winning timeout
				wintime: 0,
				// animation handling
				AnimStep: function() {
					if (this.jumping) {
						this.animSeq = 5;
					}
					else if (this.falling) {
						this.animSeq = 6;
					}
					else if (this.walking && tmr.walk.Tick()) {
						if (this.animSeq < 1 || this.animSeq > 4) {
							this.animSeq = 1;
						}
						else {
							this.animSeq++;
							if (this.animSeq > 4)
								this.animSeq = 1;
						}
					}
					else if (!this.walking) {
						this.animSeq = 0;
					}
				},
				// player drawing
				Draw: function() {
					if (this.killed) return;
					var playerY = this.y - camera_y;
					if (playerY >= -48 && playerY <= scene_h) {
						Draw.Tile(gfx_blobby, this.x, playerY + scene_y, 48, 48, this.animSeq * 48, (this.facing === RIGHT) ? 0 : 48);
					}
				},
				// player logic
				Logic: function() {
					// nothing to do if player is killed
					if (this.killed) return;
					if (this.winning) {
						// winning timeout
						if (this.wintime >= 0) {
							this.wintime++;
							if (this.wintime > 30) {
								Popup.SetState((stage === Assets.GetLevelCount()) ? 4 : 3);
								activeTransitions.push(new Transition.ShowPopup());
								this.wintime = -1;
							}
						}
						// start a happy jumping frenzy if player is winning
						if (!this.falling && !this.jumping) {
							this.jumping = true;
							this.jumpvel = Math.floor(Math.random() * 2) + 3;
						}
					}
					// apply next command
					if (!this.winning && this.command !== CMD_NONE) {
						// slight delay between command is taken and executed
						if (this.cmdDelay < ((this.command === CMD_JUMP) ? 17 : 12)) {
							this.cmdDelay++;
						}
						else {
							// execute command
							if (this.command === CMD_GOLEFT) {
								this.facing = LEFT;
							}
							else if (this.command === CMD_GORIGHT) {
								this.facing = RIGHT;
							}
							else if (this.command === CMD_JUMP) {
								if (!this.falling) {
									this.jumping = true;
									this.jumpvel = 7;
								}
							}
							else if (this.command === CMD_WAIT) {
								this.walking = false;
								this.AnimStep();
								this.waitseq = 0;
							}
							this.command = CMD_NONE;
						}
					}
					if (!this.winning) {
						// control walking
						if (this.walking) {
							if (this.facing === LEFT) {
								this.x -= 2;
								var col = this.Collision(LEFT);
								if (col == COLLISION_EDGE || col == COLLISION_WALL)
									this.walking = false;
								this.AnimStep();
							}
							else if (this.facing === RIGHT) {
								this.x += 2;
								var col = this.Collision(RIGHT);
								if (col == COLLISION_EDGE || col == COLLISION_WALL)
									this.walking = false;
								this.AnimStep();
							}
						}
						// standing
						else {
							if (this.waitseq < 80)
								this.waitseq++;
							else
								this.walking = true;
						}
					}
					// jumping
					if (this.jumping) {
						this.y -= this.jumpvel;
						if (tmr.gravity.Tick()) {
							if (this.jumpvel > 0)
								this.jumpvel--;
							else {
								this.jumping = false;
								this.falling = true;
							}
						}

						var col = this.Collision(UP);

						if (col == COLLISION_WALL || col == COLLISION_EDGE) {
							this.jumping = false;
							this.falling = true;
						}
						this.AnimStep();
					}
					// start falling
					if (!this.jumping && !this.falling) {
						this.falling = true;
						this.fallvel = 1;
						tmr.gravity.Reset();
					}
					// control falling
					if (this.falling) {
						this.y += this.fallvel;
						if (tmr.gravity.Tick() && this.fallvel < 8)
							this.fallvel++;

						if (this.Collision(DOWN) == COLLISION_WALL) {
							this.falling = false;
						}
					}
					if (!this.winning) {
						// follow player with camera
						var nexty = this.y + (48 - scene_h) / 2;
						var maxy = grid.GetH() * 48 - scene_h;
						camera_y = (nexty < 0) ? 0 : ((nexty < maxy) ? nexty : maxy);
					}
				},
				// retreive player's AABB
				GetAABB: function() {
					return new AABB(this.x + this.bbox.x, this.y + this.bbox.y,
						this.x + this.bbox.x + this.bbox.w, this.y + this.bbox.y + this.bbox.h);
				},
				// collision detection
				Collision: function(direction) {
					// get player's AABB
					var playerAABB = this.GetAABB();
					// level edges collision detection
					var lvl_realw = 480;
					var lvl_realh = grid.GetH() * 48;

					switch(direction) {
						case UP:
							if (this.y + this.bbox.y < 0) {
								this.y = -this.bbox.y;
								return COLLISION_EDGE;
							}
							break;
						case DOWN:
							if (this.y + this.bbox.y + this.bbox.h > lvl_realh) {
								this.y = lvl_realh - this.bbox.h - this.bbox.y;
								return COLLISION_EDGE;
							}
							break;
						case LEFT:
							if (this.x + this.bbox.x < 0) {
								this.x = -this.bbox.x;
								return COLLISION_EDGE;
							}
							break;
						case RIGHT:
							if (this.x + this.bbox.x + this.bbox.w > lvl_realw) {
								this.x = lvl_realw - this.bbox.w - this.bbox.x;
								return COLLISION_EDGE;
							}
							break;
					}

					// player position on grid
					var c_x = Math.floor(this.x / 48);
					var c_y = Math.floor(this.y / 48);

					// walls collision detection
					var lvlw = 10;
					var lvlh = grid.GetH();

					// only look for walls in the immediate surroundings of the player
					var from_x = Math.max(c_x - 1, 0);
					var to_x = Math.min(c_x + 3, lvlw);
					var from_y = Math.max(c_y - 1, 0);
					var to_y = Math.min(c_y + 4, lvlh);

					for (var y = from_y; y < to_y; y++) {
						for (var x = from_x; x < to_x; x++) {
							if (grid.Get(x, y).wall > 0) {

								var wall_realx = x * 48;
								var wall_realy = y * 48;

								if (playerAABB.Intersects(new AABB(wall_realx, wall_realy, wall_realx + 47, wall_realy + 47))) {
									// detected collision
									// adjust player position
									switch (direction) {
										case UP:
											this.y = wall_realy + 48 - this.bbox.y;
											break;
										case DOWN:
											this.y = wall_realy - this.bbox.h - this.bbox.y;
											break;
										case LEFT:
											this.x = wall_realx + 48 - this.bbox.x;
											break;
										case RIGHT:
											this.x = wall_realx - this.bbox.w - this.bbox.x;
											break;
									}
									// return with collision type
									return COLLISION_WALL;
								}

							}
						}
					}

					// commands collision detection
					var ci = cmdPoints.length;
					if (ci > 0) {
						while (ci--) {
							var cmdPoint = cmdPoints[ci];
							var cmdx = cmdPoint.x * 48;
							var cmdy = cmdPoint.y * 48;
							if (playerAABB.Intersects(new AABB(cmdx + 10, cmdy + 10, cmdx + 38, cmdy+ 38))) {
								this.command = cmdPoint.command;
								this.cmdDelay = 0;
								// remove command
								cmdPoints.splice(ci, 1);
								// return with collision type
								return COLLISION_COMMAND;
							}
						}
					}

					// no collisions
					return COLLISION_NONE;
				},
				// reset player
				Reset: function() {
					this.x = this.origX;
					this.y = this.origY;
					this.facing = this.origFacing;
					this.animSeq = 0;
					this.fallvel = 0;
					this.jumpvel = 0;
					this.command = CMD_NONE;
					this.cmdDelay = 0;
					this.waitSeq = 0;
					this.walking = true;
					this.falling = false;
					this.jumping = false;
					this.killed = false;
					this.winning = false;
					this.wintime = 0;
				},
				// kill the player
				Kill: function() {
					if (!this.killed) {
						this.killed = true;
						// add an fx
						effects.push(new FX(
							this.x, this.y, [[0, 0], [1, 0], [2, 0], [3, 0]]
						));
					}
				},
				// win
				Win: function() {
					if (!this.winning) {
						this.winning = true;
						this.wintime = 0;
					}
				}
			};

			// friend
			var Friend = {
				// properties
				x: 0,
				y: 0,
				facing: LEFT,
				type: 0,

				// winning animation
				animSeq: 0,
				yoffset: 0,
				jumping: false,
				jumpvel: 0,
				falling: false,
				fallvel: 0,
				friendtmr: new Timer(2),

				AnimStep: function() {
					// set animation
					if (this.jumping)
						this.animSeq = 1;
					else if (this.falling)
						this.animSeq = 2;
					else
						this.animSeq = 0;
				},

				// retreive friend's AABB
				GetAABB: function() {
					return new AABB(this.x, this.y, this.x + 48, this.y + 48);
				},

				// draw function
				Draw: function() {
					var friendY = this.y - camera_y - this.yoffset;
					if (friendY >= -48 && friendY <= scene_h) {
						var tilex = this.animSeq * 48;
						var tiley = ((this.facing === RIGHT) ? 0 : 48) + this.type * 96;
						Draw.Tile(gfx_friends, this.x, friendY + scene_y, 48, 48, tilex, tiley);
					}
				},

				// logic function
				Logic: function() {
					if (Player.winning) {
						this.friendtmr.Run();
						if (!this.falling && !this.jumping) {
							this.jumping = true;
							this.jumpvel = Math.floor(Math.random() * 2) + 3;
							this.friendtmr.Reset();
						}
						if (this.jumping) {
							this.yoffset += this.jumpvel;
							if (this.friendtmr.Tick()) {
								if (this.jumpvel > 0)
									this.jumpvel--;
								else {
									this.jumping = false;
									this.falling = true;
								}
							}

							if (this.yoffset > 96) {
								this.jumping = false;
								this.falling = true;
							}
							this.AnimStep();
						}
						if (!this.jumping && !this.falling) {
							this.falling = true;
							this.fallvel = 1;
							this.friendtmr.Reset();
						}
						if (this.falling) {
							this.yoffset -= this.fallvel;
							if (this.friendtmr.Tick() && this.fallvel < 8)
								this.fallvel++;
							if (this.yoffset <= 8) {
								this.falling = false;
								this.yoffset = 0;
							}
							this.AnimStep();
						}
					}
					else if (this.GetAABB().Intersects(Player.GetAABB())) {
						Player.Win();
					}
				},
				// reset friend
				Reset: function() {
					this.animSeq = 0;
					this.yoffset = 0;
					this.jumping = false;
					this.jumpvel = 0;
					this.falling = false;
					this.fallvel = 0;
				}
			};

			// scene objects
			var SceneObject = {
				// animated object
				Animation: function(p, c) {
					var animSeq = Math.floor(Math.random() * c.tiles.length);

					this.Draw = function() {
						// draw animation tile
						var tile = c.tiles[animSeq];
						var objx = p.x;
						var objy = p.y - camera_y;
						var objw = c.w * 48;
						var objh = c.h * 48;
						// check if object is within drawable surface
						if (objy >= -objh && objy <= scene_h)
							Draw.Tile(gfx_objects, objx, objy + scene_y, objw, objh, tile[0] * 48, tile[1] * 48);
					};

					this.Logic = function() {
						if (tmr.anim.Tick()) {
							animSeq++;
							if (animSeq > c.tiles.length - 1)
								animSeq = 0;
						}
					};
				},
				// spikes
				Spikes: function(p, c, a) {
					// spike facing direction
					var facing = DirectionFromString(a.facing);

					var aabb = (function() {
						switch (facing) {
							case UP:
								return new AABB(p.x + 6, p.y + 34, p.x + c.w * 48 - 6, p.y + c.h * 48);
							case DOWN:
								return new AABB(p.x + 6, p.y, p.x + c.w * 48 - 6, p.y + c.h * 48 - 34);
							case LEFT:
								return new AABB(p.x + 34, p.y + 6, p.x + c.w * 48, p.y + c.h * 48 - 6);
							case RIGHT:
								return new AABB(p.x, p.y + 6, p.x + c.w * 48 - 34, p.y + c.h * 48 - 6);
						}
					})();

					this.Draw = function() {
						// draw spikes
						var tile = c.tiles[facing];
						var objx = p.x;
						var objy = p.y - camera_y;
						var objw = c.w * 48;
						var objh = c.h * 48;
						if (objy >= -objh && objy <= scene_h)
							Draw.Tile(gfx_objects, objx, objy + scene_y, objw, objh, tile[0] * 48, tile[1] * 48);
					};

					this.Logic = function() {
						if (running) {
							if (Player.GetAABB().Intersects(aabb)) {
								Player.Kill();
							}
						}
					};
				},
				// lava
				Lava: function(p, c) {
					var animSeq = Math.floor(Math.random() * c.tiles.length);
					var aabb = new AABB(p.x, p.y + 8, p.x + c.w * 48, p.y + c.h * 48);

					this.Draw = function() {
						// draw tile
						var tile = c.tiles[animSeq];
						var objx = p.x;
						var objy = p.y - camera_y;
						var objw = c.w * 48;
						var objh = c.h * 48;
						if (objy >= -objh && objy <= scene_h)
							Draw.Tile(gfx_objects, objx, objy + scene_y, objw, objh, tile[0] * 48, tile[1] * 48);
					};

					this.Logic = function() {
						if (tmr.slowanim.Tick()) {
							animSeq++;
							if (animSeq > c.tiles.length - 1)
								animSeq = 0;
						}
						if (running) {
							if (Player.GetAABB().Intersects(aabb)) {
								Player.Kill();
							}
						}
					};
				}
			};

			// game objects
			var GameObject = {
				torch: function(p) { 
					return new SceneObject.Animation(p, {
						w: 1,
						h: 1 ,
						tiles: [[0, 0], [1, 0], [2, 0], [3, 0]]
					});
				},
				spikes: function(p, a) {
					return new SceneObject.Spikes(p, {
						w: 1,
						h: 1,
						tiles: [[0, 1], [1, 1], [2, 1], [3, 1]]
					}, a);
				},
				lava: function(p) {
					return new SceneObject.Lava(p, {
						w: 1,
						h: 1,
						tiles: [[0, 2], [1, 2], [2, 2]]
					});
				}
			};
			
			// scene enemy
			var SceneEnemy = {
				// ghost enemy
				Ghost: function(p, c, a) {
					var x = p.x;
					var y = p.y;
					var yoff = 0;
					var frequency = (typeof a.frequency == 'undefined') ? 60 : parseInt(a.frequency);
					var amplitude = (typeof a.amplitude == 'undefined') ? 40 : parseInt(a.amplitude);
					var xval = (typeof a.xval == 'undefined') ? 0 : parseInt(a.xval);

					var facing = DirectionFromString(a.facing);

					var animSeq = 0;
					
					// draw enemy
					this.Draw = function() {
						if (!running) return;
						var tile = c.tiles[((facing === LEFT) ? 0 : 2) + animSeq];
						var objx = x;
						var objy = yoff + y - camera_y;
						var objw = c.w * 48;
						var objh = c.h * 48;
						if (objy >= -objh && objy <= scene_h + objh) {
							Draw.Tile(gfx_enemies, objx, objy + scene_y, objw, objh, tile[0] * 48, tile[1] * 48);
						}
					};
					// do enemy logic
					this.Logic = function() {
						// animation sequence
						if (tmr.anim.Tick()) {
							animSeq++;
							if (animSeq > 1)
								animSeq = 0;
						}
						// move enemy
						xval++;
						yoff = Math.floor(Math.sin(xval / frequency) * amplitude);
						if (facing === LEFT) {
							x -= 2;
							if (x <= -(c.w * 48))
								x = p.x;
						}
						else if (facing === RIGHT) {
							x += 2;
							if (x >= 480)
								x = p.x;
						}
						// check for collisions with player
						var aabb = new AABB(x + 8, y + yoff + 36, x + c.w * 48 - 8, y + yoff + c.h * 48);
						if (Player.GetAABB().Intersects(aabb)) {
							Player.Kill();
						}
					};
					// reset enemy to original state
					this.Reset = function() {
						x = p.x;
						y = p.y;
						xval = (typeof a.xval == 'undefined') ? 0 : parseInt(a.xval);
						animSeq = 0;
					};
				},
				// spear enemy
				Spear: function(p, c, a) {
					var facing = DirectionFromString(a.facing);

					var seq = 48;
					var step = 0;

					this.Draw = function() {
						var tile = c.tiles[((facing === DOWN) ? 0 : 1)];
						var objx = p.x;
						var objy = p.y - camera_y;
						var objw = c.w * 48;
						var objh = c.h * 48;
						if (objy >= -objh && objy <= scene_h + objh) {
							if (facing === DOWN) {
								Draw.Tile(gfx_enemies, objx, objy + scene_y, objw, objh, tile[0] * 48, tile[1] * 48 + seq);
							}
							else {
								Draw.Tile(gfx_enemies, objx, objy + scene_y, objw, objh, tile[0] * 48, tile[1] * 48 - seq);
							}
						}
					};

					this.Logic = function() {
						// move the spear
						if (step == 0) {
							seq--;
							if (seq == 0)
								step = 1;
						}
						else if (step == 1) {
							seq++;
							if (seq == 48)
								step = 0;
						}
						// check for collisions with player
						var aabb = (facing === DOWN)
							? new AABB(p.x + 16, p.y, p.x + c.w * 48 - 16, p.y + c.h * 48 - seq - 20)
							: new AABB(p.x + 16, p.y + seq + 20, p.x + c.w * 48 - 16, p.y + c.h * 48);
						if (Player.GetAABB().Intersects(aabb)) {
							Player.Kill();
						}
					};

					this.Reset = function() {
						direction = (facing === DOWN) ? UP : DOWN;
						seq = 48;
						step = 0;
					};
				},
				// walking enemy
				Walking: function(p, c, a) {
					var x = p.x;
					var y = p.y;
					var facing = DirectionFromString(a.facing);

					var animSeq = 0;

					this.Draw = function() {
						var tile = c.tiles[((facing === LEFT) ? 0 : 2) + animSeq];
						var objx = x;
						var objy = y - camera_y;
						var objw = c.w * 48;
						var objh = c.h * 48;
						if (objy >= -objh && objy <= scene_h + objh) {
							Draw.Tile(gfx_enemies, objx, objy + scene_y, objw, objh, tile[0] * 48, tile[1] * 48);
						}
					};

					this.Logic = function() {
						if (tmr.anim.Tick()) {
							animSeq = (animSeq == 0) ? 1 : 0;
						}
						// move
						if (facing === RIGHT) {
							x++;
						}
						else if (facing === LEFT) {
							x--;
						}
						// guide movements
						if (x % 48 == 0) {
							var ix = Math.floor(x / 48);
							var iy = Math.floor(y / 48);
							if (facing === RIGHT) {
								ix++;
							}
							if (facing === LEFT) {
								ix--;
							}
							if (ix >= 0 && ix <= 9 && grid.Get(ix, iy).wall > 0 ||
								ix < 0 || ix > 9 ||
								iy <= grid.GetH() && grid.Get(ix, iy + 1).wall == 0)
								facing = (facing === RIGHT) ? LEFT : RIGHT;
						}
						// check for collisions with player
						var aabb = new AABB(x + 16, y + 10, x + c.w * 48 - 16, y + c.h * 48);
						if (Player.GetAABB().Intersects(aabb)) {
							Player.Kill();
						}
					};

					this.Reset = function() {
						facing = DirectionFromString(a.facing);
						x = p.x;
						y = p.y;
						animSeq = 0;
					};
				}
			};

			// game enemy
			var GameEnemy = {
				ghost: function(p, a) {
					return new SceneEnemy.Ghost(p, {
						w: 1,
						h: 2,
						tiles: [[0, 0], [1, 0], [2, 0], [3, 0]]
					}, a);
				},
				spear: function(p, a) {
					return new SceneEnemy.Spear(p, {
						w: 1,
						h: 2,
						tiles: [[0, 3], [1, 3]]
					}, a);
				},
				gnome: function(p, a) {
					return new SceneEnemy.Walking(p, {
						w: 1,
						h: 1,
						tiles: [[0, 6], [1, 6], [2, 6], [3, 6]]
					}, a);
				},
				ghoul: function(p, a) {
					return new SceneEnemy.Walking(p, {
						w: 1,
						h: 1,
						tiles: [[0, 7], [1, 7], [2, 7], [3, 7]]
					}, a);
				}
			};
			
			// game object collection
			var objects = [];

			// game enemy collection
			var enemies = [];

			// running flag
			var running = false;

			// press event
			var PressEvent = function(e, agent) {
				if (e.preventDefault) e.preventDefault();
				if (e.stopPropagation) e.stopPropagation();
				if (playMode || activeTransitions.length > 0
					|| Popup.IsActive()) return false;
				var coords = TranslateCoords(e, agent);
				if (PointInRect(coords.x, coords.y, 0, scene_y, 480, 480)) {
					if (editState == -1) {
						// dragging start
						dragging = true;
						origin_y = coords.y;
						origin_c = camera_y;
						if (agent === 'mouse')
							canvas.style.cursor = 'move';
					}
					else {
						// place or clear command
						var mapx = Math.floor(coords.x / 48);
						var mapy = Math.floor((coords.y - scene_y + camera_y) / 48);
						var tileData = grid.Get(mapx, mapy);
						if (editState < 4)  {
							if (tileData.wall == 0) {
								tileData.command = editState;
								editState = -1;
								EditPanel.Refresh();
							}
						}
						else if (editState == 4)
							tileData.command = -1;
					}
				}
				return false;
			};

			// move event
			var MoveEvent = function(e, agent) {
				if (e.preventDefault) e.preventDefault();
				if (e.stopPropagation) e.stopPropagation();
				if (playMode || activeTransitions.length > 0
					|| Popup.IsActive()) return false;
				if (dragging) {
					// drag camera
					var coords = TranslateCoords(e, agent);
					var camera_max = Math.abs(48 * grid.GetH() - 480);
					var camera_new_y = origin_c - (coords.y - origin_y);
					if (camera_new_y < 0)
						camera_new_y = 0;
					else if (camera_new_y > camera_max)
						camera_new_y = camera_max;
					camera_y = camera_new_y;
				}
				return false;
			};

			// release event
			var ReleaseEvent = function(e, agent) {
				if (e.preventDefault) e.preventDefault();
				if (playMode || activeTransitions.length > 0
					|| Popup.IsActive()) return false;
				if (dragging) {
					dragging = false;
					if (agent === 'mouse')
						canvas.style.cursor = 'auto';
				}
				return false;
			};

			return {
				// initialize game scene
				Init: function() {
					// add event handlers
					canvas.addEventListener('mousedown',  function(e) { PressEvent(e, 'mouse');   }, false);
					canvas.addEventListener('touchstart', function(e) { PressEvent(e, 'touch');   }, false);
					window.addEventListener('mousemove',  function(e) { MoveEvent(e, 'mouse');    }, false);
					window.addEventListener('touchmove',  function(e) { MoveEvent(e, 'touch');    }, false);
					window.addEventListener('mouseup',    function(e) { ReleaseEvent(e, 'mouse'); }, false);
					window.addEventListener('touchend',   function(e) { ReleaseEvent(e, 'touch'); }, false);
				},
				// draw function
				Draw: function() {
					if (!grid)
						return;
					// draw backgrounds, walls and command points
					var toY = (playMode) ? 13 : 11;
					for (var y = 0; y < toY; y++) {
						for (var x = 0; x < 10; x++) {
							var posx = x * 48;
							var posy = y * 48 - (camera_y % 48) + scene_y;

							var ix = x;
							var iy = y + Math.floor(camera_y / 48);

							if (ix < 0 || iy < 0 || ix > grid.GetW() - 1 || iy > grid.GetH() - 1)
								continue;

							var tileData = grid.Get(ix, iy);

							if (tileData.wall == 0 && tileData.background > 0) {
								var tilerow = Math.floor((tileData.background - 1) / 10);
								var tilecol = (tileData.background - 1) % 10;
								Draw.Tile(gfx_master, posx, posy, 48, 48, tilecol * 48, tilerow * 48);
							}
							if (tileData.wall > 0) {
								var tilerow = Math.floor((tileData.wall - 1) / 10);
								var tilecol = (tileData.wall - 1) % 10;
								Draw.Tile(gfx_master, posx, posy, 48, 48, tilecol * 48, tilerow * 48);
							}
						}
					}
					// draw objects
					for (var i = 0; i < objects.length; i++)
						objects[i].Draw();
					// draw blobby
					Player.Draw();
					// draw friend
					Friend.Draw();
					// draw enemies
					for (var i = 0; i < enemies.length; i++)
						enemies[i].Draw();
					// draw effects
					if (running) {
						// draw effects
						for (var i = 0; i < effects.length; i++)
							effects[i].Draw();
					}
					// draw command points
					if (running) {
						for (var i = 0; i < cmdPoints.length; i++) {
							var cmdPoint = cmdPoints[i];
							var pointY = cmdPoint.y * 48 - camera_y;
							if (pointY >= 0 && pointY <= scene_h)
								Draw.Tile(gfx_icons, cmdPoint.x * 48, pointY + scene_y, 48, 48, cmdPoint.command * 48, 0);
						}
					}
					else {
						for (var y = 0; y < toY; y++) {
							for (var x = 0; x < 10; x++) {
								var posx = x * 48;
								var posy = y * 48 - (camera_y % 48) + scene_y;

								var ix = x;
								var iy = y + Math.floor(camera_y / 48);

								if (ix < 0 || iy < 0 || ix > grid.GetW() - 1 || iy > grid.GetH() - 1)
									continue;

								var tileData = grid.Get(ix, iy);

								if (tileData.command > -1)
									Draw.Tile(gfx_icons, posx, posy, 48, 48, tileData.command * 48, 0);
							}
						}
					}

				},
				// logic function
				Logic: function() {
					// run timers
					Iterate(tmr, function(key, timer) {
						timer.Run();
					});
					// object logic
					for (var i = 0; i < objects.length; i++)
						objects[i].Logic();
					// game logic
					if (running) {
						// player logic
						Player.Logic();
						// friend logic
						Friend.Logic();
						// enemies logic
						for (var i = 0; i < enemies.length; i++)
							enemies[i].Logic();
						// effects logic
						var fi = effects.length;
						while (fi--) {
							var fx = effects[fi];
							if (fx.active)
								fx.Logic();
							else
								effects.splice(fi, 1);
						}
					}
				},
				// load level
				LoadLevel: function(level) {
					// clear existing level
					if (grid != null) {
						delete grid;
						grid = null;
					}
					// map constraints
					var mapw = level.width;
					var maph = level.height;
					if (mapw !== 10) return;
					// create a new level
					grid = new Grid(maph);
					// inspect level layers
					if (level.layers.length !== 5) return;
					var lyBackground, lyWalls, lyObjects, lyEnemies, lyEntities;
					// get references to layers so we can import level data directly
					for (var i = 0; i < level.layers.length; i++) {
						var layer = level.layers[i];
						var layerName = layer.name.toLowerCase();
						if (layerName === 'background' && layer.type === 'tilelayer')
							lyBackground = layer;
						else if (layerName === 'walls' && layer.type === 'tilelayer')
							lyWalls = layer;
						else if (layerName === 'objects' && layer.type === 'objectgroup')
							lyObjects = layer;
						else if (layerName === 'enemies' && layer.type === 'objectgroup')
							lyEnemies = layer;
						else if (layerName === 'entities' && layer.type === 'objectgroup')
							lyEntities = layer;
					}
					// check if all layers are found
					if (typeof lyBackground == 'undefined' ||
						typeof lyWalls == 'undefined'      ||
						typeof lyObjects == 'undefined'    ||
						typeof lyEnemies == 'undefined'     ||
						typeof lyEntities == 'undefined')
						return;
					// extract tile data
					for (var mapy = 0; mapy < maph; mapy++) {
						for (var mapx = 0; mapx < mapw; mapx++) {
							var dataIndex = mapx + mapy * mapw;
							grid.Set(mapx, mapy, new TileData(
								lyBackground.data[dataIndex],
								lyWalls.data[dataIndex]
							));
						}
					}
					// clear object data
					objects = [];
					// extract object data
					var objCount = lyObjects.objects.length;
					for (var i = 0; i < objCount; i++) {
						var obj = lyObjects.objects[i];
						// check if object type is in collection of available game objects
						var GenerateObj = GameObject[obj.type.toLowerCase()];
						if (GenerateObj instanceof Function) {
							// create a new game object
							var gameObj = GenerateObj({
								x: obj.x,
								y: obj.y
							}, obj.properties);
							// push the new object into objects array
							objects.push(gameObj);
						}
					}
					// clear enemy data
					enemies = [];
					// extract enemies data
					var enemyCount = lyEnemies.objects.length;
					for (var i = 0; i < enemyCount; i++) {
						var enemy = lyEnemies.objects[i];
						// check if enemy type is in collection of available game enemies
						var GenerateEnemy = GameEnemy[enemy.type.toLowerCase()];
						if (GenerateEnemy instanceof Function) {
							// create a new game enemy
							var gameEnemy = GenerateEnemy({
								x: enemy.x,
								y: enemy.y
							}, enemy.properties);
							// push the new enemy into enemies array
							enemies.push(gameEnemy);
						}
					}
					// reset entities data
					Player.x = Player.origX = 0;
					Player.y = Player.origY = 0;
					Friend.x = 0;
					Friend.y = 0;
					// extract entities data
					var entCount = lyEntities.objects.length;
					for (var i = 0; i < entCount; i++) {
						var ent = lyEntities.objects[i];
						// check if object is player
						if (ent.type === 'blobby') {
							Player.x = Player.origX = ent.x;
							Player.y = Player.origY = ent.y;
							Player.origFacing = Player.facing = DirectionFromString(ent.properties.facing);
						}
						// check if object is friend
						else if (ent.type === 'friend') {
							Friend.x = ent.x;
							Friend.y = ent.y;
							Friend.facing = DirectionFromString(ent.properties.facing);
							Friend.type = parseInt(ent.properties.type, 10);
							// set HUD friend
							HUD.friend = Friend.type;
						}
					}
					// reset everything
					camera_y = 0; 
					// reset timers
					Iterate(tmr, function(key, timer) {
						timer.Reset();
					});
				},
				// sets play mode
				SetPlayMode: function(_playMode) {
					playMode = _playMode;
					scene_h = (playMode) ? 576 : 480;
				},
				// get play mode state
				GetPlayMode: function() {
					return playMode;
				},
				// set camera y
				SetCameraY: function(cameraY) {
					camera_y = cameraY;
				},
				// get camera y
				GetCameraY: function() {
					return camera_y;
				},
				// run game
				Run: function() {
					running = true;
					// create command points
					cmdPoints = [];
					var mapw = 10;
					var maph = grid.GetH();
					for (var y = 0; y < maph; y++) {
						for (var x = 0; x < mapw; x++) {
							var tileData = grid.Get(x, y);
							if (tileData.command > -1)
								cmdPoints.push({
									x: x,
									y: y,
									command: tileData.command
								});
						}
					}
					// reset game
					this.Reset();
					// update HUD
					HUD.game_mode = 1;
				},
				// stop game
				Stop: function() {
					running = false;
					// update HUD
					HUD.game_mode = 0;
				},
				// reset game scene
				Reset: function() {
					Player.Reset();
					Friend.Reset();
					// reset enemies
					for (var i = 0; i < enemies.length; i++) {
						enemies[i].Reset();
					}
					// clear the effects array
					effects = [];
				}
			};
		})();

		// editing panel
		var EditPanel = {
			Buttons: [ el.btnGoLeft, el.btnGoRight, el.btnJump, el.btnWait, el.btnErase ],
			Refresh: function() {
				for (var i = 0; i < this.Buttons.length; i++)
					this.Buttons[i].SetActive(editState == i);
			},
			Logic: function() {
				if (el.btnGoLeft.UserEvent()) {
					editState = (editState == 0) ? -1 : 0;
					this.Refresh();
				}
				else if (el.btnGoRight.UserEvent()) {
					editState = (editState == 1) ? -1 : 1;
					this.Refresh();
				}
				else if (el.btnJump.UserEvent()) {
					editState = (editState == 2) ? -1 : 2;
					this.Refresh();
				}
				else if (el.btnWait.UserEvent()) {
					editState = (editState == 3) ? -1 : 3;
					this.Refresh();
				}
				else if (el.btnErase.UserEvent()) {
					editState = (editState == 4) ? -1 : 4;
					this.Refresh();
				}
			},
			Move: function(dy) {
				for (var i = 0; i < this.Buttons.length; i++)
					this.Buttons[i].GetProps().y += dy;
			},
			Enable: function(enabled) {
				for (var i = 0; i < this.Buttons.length; i++)
					this.Buttons[i].SetEnabled(enabled);
			}
		};

		// enter edit mode
		var EnterEditMode = function() {
			// prepare scene
			Scene.SetPlayMode(false);
			// configure buttons
			el.btnStop.SetEnabled(false);
			el.btnStart.SetEnabled(true);
			EditPanel.Enable(true);
		};

		// enter game mode
		var EnterGameMode = function() {
			// prepare scene
			Scene.SetPlayMode(true);
			// configure buttons
			el.btnStop.SetEnabled(true);
			el.btnStart.SetEnabled(false);
			EditPanel.Enable(false);
		};

		// reset controls
		var SetupEditMode = function() {
			EditPanel.Move(-144);
			editState = -1;
			EditPanel.Refresh();
			el.btnStop.GetProps().y -= 48;
			el.btnStart.GetProps().y -= 48;
		};

		return {
			Init: function(_canvas) {
				// canvas reference
				canvas = _canvas;
				// get configuration from localstorage
				LoadConfig();
				// set active stage to last unlocked stage
				if (typeof config.unlocked != 'undefined' &&
					config.unlocked >= 1 && config.unlocked <= 42)
					stage = config.unlocked;
				// get graphic assets
				gfx_ui = Assets.Get('gfx_ui');
				gfx_icons = Assets.Get('gfx_icons');
				gfx_labels = Assets.Get('gfx_labels');
				gfx_num = Assets.Get('gfx_num');
				gfx_master = Assets.Get('gfx_master');
				gfx_objects = Assets.Get('gfx_objects');
				gfx_blobby = Assets.Get('gfx_blobby');
				gfx_fx = Assets.Get('gfx_fx');
				gfx_friends = Assets.Get('gfx_friends');
				gfx_hudlabels = Assets.Get('gfx_hudlabels');
				gfx_enemies = Assets.Get('gfx_enemies');
				// initialize ui elements
				Iterate(el, function(key, element) {
					element.Init();
				});
				// initialize scene
				Scene.Init();
				// initialize popup screen
				Popup.Init();
				// load the starting stage
				Scene.LoadLevel(Assets.GetLevel(stage));
				// set editing mode
				EnterEditMode();
				if (stage === 1) {
					// popup story on start
					Popup.SetState(0);
					activeTransitions.push(new Transition.ShowPopup());
				}
				// start music playback
				if (Music.HasOggSupport()) {
					// add tracks to player
					Music.SetTracks(Assets.GetTracks());
					// start playback
					if (config.music === true)
						Music.StartPlayback();
				}
				else {
					// show the disabled icon on the music button
					el.btnMusic.SetActive(false);
				}
			},
			DoDrawing: function() {
				// draw the scene
				Scene.Draw();
				// draw the backgrounds
				Draw.Tile(gfx_ui, 0, 0, 480, 96, 0, 0);
				// draw the interface
				Iterate(el, function(key, element) {
					element.Draw();
				});
				// draw heads up display
				HUD.Draw();
				// stage number
				var stagex = (stage < 10) ? 42 : 30;
				Draw.Num(gfx_num, stage, stagex, 20);
				// music disabled overlay icon
				if (!config.music)
					Draw.Tile(gfx_icons, 312, 10, 48, 48, 336, 0);
				//popup
				Popup.Draw();
			},
			DoLogic: function() {
				if (!Popup.IsActive()) {
					// scene logic
					Scene.Logic();

					// edit panel logic
					EditPanel.Logic();
				}

				// button user events
				if (el.btnStage.UserEvent()) {
					if (Popup.GetState() === 1) {
						Popup.SetState(-1);
						Popup.SetActive(false);
					}
					else {
						Popup.SetState(1);
						if (!Popup.IsActive()) {
							activeTransitions.push(new Transition.ShowPopup());
						}
					}
				}
				else if (el.btnMusic.UserEvent()) {
					if (Music.HasOggSupport()) {
						config.music = !config.music;
						if (config.music)
							Music.StartPlayback();
						else
							Music.StopPlayback();
						SaveConfig();
					}
				}
				else if (el.btnInfo.UserEvent()) {
					if (Popup.GetState() === 2) {
						Popup.SetState(-1);
						Popup.SetActive(false);
					}
					else {
						Popup.SetState(2);
						if (!Popup.IsActive()) {
							activeTransitions.push(new Transition.ShowPopup());
						}
					}
				}
				if (el.btnStart.UserEvent()) {
					// enter game mode
					EnterGameMode();
					// begin start stage transition
					activeTransitions.push(new Transition.ScrollCameraUp());
					activeTransitions.push(new Transition.MoveEditPanel());
					activeTransitions.push(new Transition.MoveStopButton());
					// wait until all transitions are finished, then run the scene
					(function Wait() {
						if (activeTransitions.length > 0)
							setTimeout(Wait, 10);
						else {
							// run scene
							Scene.Run();
						}
					})();
				}
				if (el.btnStop.UserEvent()) {
					// stop game
					Scene.Stop();
					// reset scene
					Scene.Reset();
					// enter editing mode
					EnterEditMode();
					// reset controls
					SetupEditMode();
				}

				// popup logic
				Popup.Logic();

				// control and run transition states
				var ti = activeTransitions.length;
				if (ti > 0) {
					while (ti--) {
						var tr = activeTransitions[ti];
						// do transition logic
						if (tr.Logic()) {
							// remove this transition from list of active transitions
							activeTransitions.splice(ti, 1);
						}
					}
				}
			},
			// unlock all levels cheat
			Cheat: function() {
				config.unlocked = Assets.GetLevelCount();
				SaveConfig();
			}
		};
	})();

	// core module
	var Core = (function() {
		// reference to canvas and render context
		var canvas, ctx;

		// track window height
		var windowHeight;

		// fallbacks for requestAnimFrame
		var RequestAnimFrame = (
			window.requestAnimationFrame       ||
			window.webkitRequestAnimationFrame ||
			window.mozRequestAnimationFrame    ||
			window.oRequestAnimationFrame      ||
			window.msRequestAnimationFrame     ||
			function (callback) {
				window.setTimeout(callback, 1000 / 60);
			}
		);

		// main game loop
		var Run = function() {
			var cw = canvas.width;
			var ch = canvas.height;

			// poll window height for changes
			var dh = window.innerHeight;
			if (dh !== windowHeight) {
				windowHeight = dh;
				
				// re-scale canvas
				var ratio = cw / ch;
				var width = dh * ratio;

				canvas.style.width = width + 'px';
				canvas.style.height = dh + 'px';
			}

			// clear canvas area
			ctx.clearRect(0, 0, cw, ch);

			// draw to context
			Game.DoDrawing();

			// logic
			Game.DoLogic();

			// loopback
			RequestAnimFrame(Run);
		};

		return {
			Start: function() {
				// retreive canvas
				canvas = document.getElementById('blobby');

				// retreive loadscreen elements
				var loadscreen = document.getElementById('blobby-loadscreen');
				var progressbar = document.getElementById('blobby-progressbar');
				var progresslbl = document.getElementById('blobby-progresslabel');

				// retreive canvas context and initialize the drawing module
				Draw.SetContext(ctx = canvas.getContext('2d'));

				// initialize the input module

				// load game assets
				Assets.Load(function() {
					// finished loading
					// remove the loadscreen and bring up game canvas
					loadscreen.style.display = 'none';
					canvas.style.display = 'block';
					// initialize game
					Game.Init(canvas);
					// run the main loop
					Run();
				}, function(items, nitems) {
					// track progress
					var progress = Math.min(100, Math.floor((items + 1) * 100 / nitems)) + '%';
					progressbar.style.width = progress;
					progresslbl.innerHTML = progress;
				});
			}
		};
	})();

	// start the game as the window loads
	window.addEventListener('load', Core.Start, false);

	// cheat
	return {
		// unlock all levels
		UnlockAll: function() {
			Game.Cheat();
		}
	};
})();
