<?php
namespace App\Http\Controllers\Dashboard;


use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Spatie\Permission\Models\Role;
use Illuminate\Validation\Rule AS ValidationRule;
use Spatie\Permission\Models\Permission;
use Propaganistas\LaravelPhone\PhoneNumber;
use Illuminate\Support\Str;
use App\Http\Controllers\API\UserController as ApiUserController;
use Carbon\Carbon;
use App\Exports\PartnersExport;
use Maatwebsite\Excel\Facades\Excel;
use App\User;
use App\Matrix;
use Storage;
use Auth;
use Hash;
use DB;


class UserController extends Controller
{
    function __construct(){
         //$this->middleware('permission:user-list|user-create|user-edit|user-delete', ['only' => ['index','store', 'show']]);
         //$this->middleware('permission:user-create', ['only' => ['create','store']]);
         //$this->middleware('permission:user-edit', ['only' => ['edit','update']]);
         //$this->middleware('permission:user-delete', ['only' => ['destroy']]);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if( $request->filled('search') ){
            $users = User::withTrashed()->orderBy('id', 'ASC')->search(Str::of($request->input('search'))->trim())->paginate(( $request->has('limit') ? (int) $request->input('limit') : 10 ));
        }else{
            if( $request->has('search') ){
                $request->offsetUnset('search');
            }
            $users = User::withTrashed()->orderBy('id','ASC')->paginate(( $request->has('limit') ? (int) $request->input('limit') : 10 ));
        }
        return view('dashboard.users.index', compact('users'));
    }

    public function partners(Request $request){
        $users = User::withTrashed()->permission('is-partner')->orderBy('uid', 'ASC');

        if( $request->filled('search') ){
            $users = $users->search(Str::of($request->input('search'))->trim());
        }
        $users = $users->get()->sortByDesc(function($user) use ($request){
            return $user->getPaymentsCountByTreeAttribute($request->all())->get('points');
        })->paginate(( $request->has('limit') && !empty($request->input('limit')) ? ( $request->input('limit') == 'all' ? (int) $users->count() : (int) $request->input('limit') ) : 10 ));
        return view('dashboard.partners.index', compact('users'));
    }
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $roles = Role::pluck('name','name')->all();
        return view('dashboard.users.create', compact('roles'));
    }

    public function create_partners(){
        $permission = Permission::where('name', 'is-partner')->first();
        $roles = Role::where('id', $permission->roles->first()->id)->pluck('name','name')->first();
        return view('dashboard.partners.create', compact('roles'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'email' => 'required|email|unique:users,email',
            'roles' => 'required'
        ]);

        if( $request->filled('phone') ){
            $phone = PhoneNumber::make($request->input('phone'), 'RU');
            $request->merge(['phone' => $phone->formatE164()]);
            $this->validate($request, [
                'phone' => 'nullable|phone:AUTO,RU|unique:users,phone'
            ]);
        }

        if( $request->filled('password') ){
            $this->validate($request, [
                'password' => 'string|confirmed'
            ]);
            $generate_password = $request->input('password');
        }else{
            $generate_password = Str::random(32);
        }

        $request->merge([
            'password' => Hash::make($generate_password),
            'api_token' => Str::random(60)
        ]);

        $user = User::create($request->all());
        $user->notify(new \App\Notifications\WelcomeUser($password = $generate_password));
        $user->assignRole($request->input('roles'));

        return redirect()->route('dashboard.users.index')->with('success', 'Создание выполнено успешно');
    }

    public function store_partners(Request $request)
    {
        $this->validate($request, [
            'lastname' => 'required',
            'firstname' => 'required',
            'email' => 'required|email|unique:users,email',
            'phone' => 'required|phone:AUTO,RU|unique:users,phone',
            'roles' => 'required'
        ]);

        if( $request->filled('phone') ){
            $phone = PhoneNumber::make($request->input('phone'), 'RU');
            $request->merge(['phone' => $phone->formatE164()]);
            $this->validate($request, [
                'phone' => 'nullable|phone:AUTO,RU|unique:users,phone',
            ]);
        }

        if( $request->filled('password') ){
            $this->validate($request, [
                'password' => 'string|confirmed'
            ]);
            $generate_password = $request->input('password');
        }else{
            $generate_password = Str::random(32);
        }

        if( $request->filled('parent_uid') ){
            if( User::where('uid', $request->input('parent_uid') )->count() == 0 ){
                return redirect()->back()->withInput()->with('error', 'Партнёр, по указанному ID - отсутствует');
            }
        }

        $request->merge([
            'password' => Hash::make($generate_password),
            'api_token' => Str::random(60)
        ]);
        
        $user = User::create($request->all());
        $user->notify(new \App\Notifications\WelcomeUser($password = $generate_password));
        $user->assignRole($request->input('roles'));

        if( $request->filled('parent_uid') ){
            $_ARRAY_TREE = ApiUserController::GetTreeChartGenerate(User::where('uid', $request->input('parent_uid') )->first());
            $_FIND_EMPTY = ApiUserController::findEmptyPositionTreeChart($_ARRAY_TREE);
            if( !empty($_FIND_EMPTY['uid']) && $_FIND_EMPTY['position'] ){
                Matrix::create([
                    'user_uid' => $user->uid,
                    'parent_uid' => (int) $_FIND_EMPTY['uid'],
                    'position' => $_FIND_EMPTY['position'],
                ]);
            }else{
                return redirect()->route('dashboard.partners.show', $user)->with('error', 'Создание выполнено успешно, но произошла ошибка при распределении партнёра');
            }
        }

        return redirect()->route('dashboard.partners.show', $user)->with('success', 'Создание выполнено успешно');
    }
    public function exportPartners($extension, Request $request){
        if( empty($extension) ){
            abort(404);
        }
        $params = collect($request->all());
        $file_name = ( $params->isEmpty() ) ? Str::slug(Carbon::now()->format('Y-m-d H:i:s'), '-') : Str::slug($params->implode('_'), '-');
        switch (Str::upper($extension)) {
            case 'XLSX':
                return Excel::download(new PartnersExport($request->all()), $file_name . '.xlsx', \Maatwebsite\Excel\Excel::XLSX);
            break;
            case 'TSV':
                return Excel::download(new PartnersExport($request->all()), $file_name . '.tsv', \Maatwebsite\Excel\Excel::TSV);
            break;
            case 'ODS':
                return Excel::download(new PartnersExport($request->all()), $file_name . '.ods', \Maatwebsite\Excel\Excel::ODS);
            break;
            case 'XLS':
                return Excel::download(new PartnersExport($request->all()), $file_name . '.xls', \Maatwebsite\Excel\Excel::XLS);
            break;
            case 'HTML':
                return Excel::download(new PartnersExport($request->all()), $file_name . '.html', \Maatwebsite\Excel\Excel::HTML);
            break;
            case 'MPDF':
                return Excel::download(new PartnersExport($request->all()), $file_name . '.pdf', \Maatwebsite\Excel\Excel::MPDF);
            break;
            case 'DOMPDF':
                return Excel::download(new PartnersExport($request->all()), $file_name . '.pdf', \Maatwebsite\Excel\Excel::DOMPDF);
            break;
            case 'TCPDF':
                return Excel::download(new PartnersExport($request->all()), $file_name . '.pdf', \Maatwebsite\Excel\Excel::TCPDF);
            break;
            case 'CSV':
            default:
                return Excel::download(new PartnersExport($request->all()), $file_name . '.csv', \Maatwebsite\Excel\Excel::CSV, [
                    'Content-Type' => 'text/csv',
                ]);
            break;
        }
    }
    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(User $user)
    {
        return view('dashboard.partners.show', compact('user'));
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(User $user)
    {
        $roles = Role::pluck('name','name')->all();
        $userRole = $user->roles->pluck('name','name')->all();
        return view('dashboard.users.edit', compact('user','roles','userRole'));
    }


    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(User $user, Request $request)
    {
        $this->validate($request, [
            'email' => 'required|email|unique:users,email,' . $user->id,
            'uid' => 'required|unique:users,uid,' . $user->id,
            'phone' => 'nullable|phone:AUTO,RU|unique:users,phone,' . $user->id,
            'roles' => 'required'
        ]);
        
        if( $request->filled('phone') ){
            $phone = PhoneNumber::make($request->input('phone'), 'RU');
            $request->merge(['phone' => $phone->formatE164()]);
            $this->validate($request, [
                'phone' => 'nullable|phone:AUTO,RU|unique:users,phone,' . $user->id,
            ]);
        }
        if( empty($user->api_token) ){
            $request->merge([
                'api_token' => Str::random(60)
            ]);
        }
        if( $request->filled('password') ){
            $this->validate($request, [
                'password' => 'string|confirmed'
            ]);
            $generate_password = $request->input('password');
            $request->merge([
                'password' => Hash::make($generate_password)
            ]);
        }else{
            $request->offsetUnset('password');
            $request->offsetUnset('password_confirmation');
        }

        if( empty($user->parent_uid) ){
            if( $request->filled('parent_uid') ){
                if( $user->uid == $request->input('parent_uid') ){
                    return redirect()->back()->withInput()->with('error', 'Вы не можете указывать свой ID как партнёра, который вас пригласил.');
                }
                $request->merge([
                    'parent_uid' => $request->input('parent_uid')
                ]);
            }else{
                $request->offsetUnset('parent_uid');
            }
        }
        $user->update($request->all());
        DB::table('model_has_roles')->where('model_id', $user->id)->delete();
        $user->assignRole($request->input('roles'));

        return redirect()->route('dashboard.users.index')->with('success', 'Обновление выполнено успешно');
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(User $user){
        $user->delete();
        return redirect()->route('dashboard.users.index')->with('success', 'Удаление выполнено успешно');
    }

    public function registration_resend(User $user){
        $generate_password = Str::random(32);
        $user->update([
            'password' => Hash::make($generate_password),
        ]);
        $user->notify(new \App\Notifications\WelcomeUser($password = $generate_password));
        return redirect()->back()->with('success', 'Сброс выполнен успешно, на E-mail адрес отправлены новые данные.');
    }

    public function restore($uid){
        if( User::onlyTrashed()->where('uid', $uid)->count() == 0 ){
            abort(404);
        }
        $user = User::onlyTrashed()->where('uid', $uid)->first();
        $user->restore();
        return redirect()->route('dashboard.users.index')->with('success', 'Восстановление выполнено успешно');
    }

    public function destroy_trash($uid){
        if( User::onlyTrashed()->where('uid', $uid)->count() == 0 ){
            abort(404);
        }
        $user = User::onlyTrashed()->where('uid', $uid)->first();
        DB::table('model_has_roles')->where('model_id', $user->id)->delete();
        $user->forceDelete();
        return redirect()->route('dashboard.users.index')->with('success', 'Полное удаление выполнено успешно');
    }
}