<?php

define('USERS_SOCIAL_TYPE_VKONTAKTE',     1);
define('USERS_SOCIAL_TYPE_FACEBOOK',      2);
define('USERS_SOCIAL_TYPE_ODNOKLASSNIKI', 4);
define('USERS_SOCIAL_TYPE_MAILRU',        8);

abstract class UsersBase extends UsersModule
{
    /** @var UsersModel */
    public $model = null;


    function init()
    {
        parent::init();

        $this->module_title = 'Пользователи';
        $this->profilePhonesLimit = 1;
    }

    /**
     * Формирование URL
     * @param string $key ключ
     * @param mixed $opts параметры
     * @param boolean $dynamic динамическая ссылка
     * @return string
     */
    public static function url($key = '', array $opts = array(), $dynamic = false)
    {
        $base = static::urlBase(LNG, $dynamic);
        switch ($key)
        {
            # Авторизация
            case 'login':
                return $base . '/user/login';
                break;
            # Выход
            case 'logout':
                return $base . '/user/logout';
                break;
            # Регистрация
            case 'register':
                return $base . '/user/register' . static::urlQuery($opts);
                break;
            # Авторизация
            case 'login.social':
                return $base . '/user/loginsocial/' . (!empty($opts['provider']) ? $opts['provider'] : '');
                break;
            # Восстановление пароля
            case 'forgot':
                return $base . '/user/forgot' . static::urlQuery($opts);
                break;
            # Ссылка активации акканута
            case 'activate':
                return $base . '/user/activate' . static::urlQuery($opts);
                break;
            # Пользовательское соглашение
            case 'agreement':
                return $base . '/agreement.html';
                break;
            # Настройки профиля
            case 'my.settings':
                return $base . '/user/settings' . static::urlQuery($opts);
                break;
            # Профиль пользователя
            case 'my.profile':
                return $base . '/user/profile' . static::urlQuery($opts);
                break;
            # Профиль пользователя: разделы
            case 'my.profile.tab': {
                return $base.'/user/profile?t='.$opts['tab'].
                    ( ! empty($opts['subtab']) ? '&st='.$opts['subtab'] : '' );
            } break;
            case 'forum': {
                $url = config::sys('forum.baseurl', false);
                return $url ? $url : $base;
            } break;
            case 'vbulletin5':{
                return $base.'/user/vbulletin5';
            }
        }
        return $urlBase;
    }

    /**
     * Доступно ли редактирование даты рождения в профиле
     * @return bool
     */
    public static function profileBirthdate()
    {
        return config::sys('users.profile.birthdate', true);
    }

    /**
     * Описание seo шаблонов страниц
     * @return array
     */
    public function seoTemplates()
    {
        $aTemplates = array(
            'pages' => array(
                'login' => array( // login
                    't'      => 'Авторизация',
                    'macros' => array(
                        // ...
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'register' => array( // register
                    't'      => 'Регистрация',
                    'macros' => array(
                        // ...
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'forgot' => array( // forgot
                    't'      => 'Восстановление пароля',
                    'macros' => array(
                        // ...
                    ),
                ),
            ),
        );

        return $aTemplates;
    }

    function cleanUserData(&$aData, $mKeys = 'all')
    {
        if($mKeys == 'all') {
            $mKeys = array_keys($aData);
        }

        foreach($mKeys as $key)
        {
            switch ($key)
            {
                case 'name': # имя
                case 'surname': # фамилия
                {
                    # допустимые символы:
                    # латиница, кирилица, тире
                    $aData[$key] = preg_replace('/[^a-zа-яёїієґ\-\s]+/iu', '', $aData[$key]);
                    $aData[$key] = mb_substr( $aData[$key], 0, 30 );
                    $aData[$key] = trim( $aData[$key] );
                } break;
                case 'birthdate': # дата рождения
                {
                    if( ! static::profileBirthdate()) break;

                    if (empty($aData[$key]) || !checkdate($aData[$key]['month'], $aData[$key]['day'], $aData[$key]['year'])){
                        $this->errors->set( _t('users', 'Дата рождения указана некорректно'), 'birthdate' );
                    } else {
                        $aData[$key] = "{$aData[$key]['year']}-{$aData[$key]['month']}-{$aData[$key]['day']}";
                    }
                } break;
                case 'site': # сайт
                {
                    $aData[$key] = mb_substr( $aData[$key], 0, 255 );
                    $aData[$key] = str_replace(array('http://','https://', 'ftp://'), '', $aData[$key]);
                } break;
                case 'phone': # телефон (один)
                {
                    if( ! isset($aData['phones']) ) {
                        # формируем на его основе 'phones'
                        $aData['phone'] = preg_replace('/[^\s\(\)\+\-0-9 ]/','',trim($aData[$key]));
                        $aPhones = array();
                        if(strlen($aData['phone'])>4) {
                            $aPhones[] = $aData['phone'];
                        }
                        $aData['phones'] = join(';', $aPhones);
                    } else {
                        unset($aData['phone']);
                    }
                } break;
                case 'phones': # телефоны
                {
                    $aTemp = array(); $limit = $this->profilePhonesLimit;
                    foreach($aData[$key] as $p)
                    {   # чистим телефон
                        $p = preg_replace('/[^\s\(\)\+\-0-9 ]/','',trim($p));
                        if(strlen($p)>4) {
                            $aTemp[] = mb_substr($p, 0, 40);
                        }
                    }
                    if($limit > 0 && sizeof($aTemp) > $limit) $aTemp = array_slice($aTemp, 0, $limit);
                    $aData[$key] = join(';', $aTemp);
                    // сохраняем первый телефон в отдельное поле
                    $aData['phone'] = ( sizeof($aTemp) > 0 ? reset($aTemp) : '' );
                } break;
                case 'skype': # skype
                {
                    $aData[$key] = preg_replace('/[^\.\s\[\]\_a-zA-Z0-9]/', '', $aData[$key]);
                } break;
                case 'icq': # icq
                {
                    $aData[$key] = preg_replace('/[^\.\-\s\[\]\_a-zA-Z0-9]/', '', $aData[$key]);
                } break;
                case 'city_id':
                {
                    if($aData['city_id'] > 0) {
                        // проверяем корректность указанного города
                        if( ! Geo::isCity( $aData['city_id'] ) ) {
                            $aData['city_id'] = 0;
                        }
                    }
                } break;
            }
        }
    }


    /**
     * Иницилизация компонента работы с соц. аккаунтами
     * @return UsersSocial
     */
    public function social()
    {
        static $i;
        if (!isset($i)) {
            $i = new UsersSocial();
        }

        return $i;
    }

    /**
     * Регистрация пользователя
     * @param array $aData данные
     * @param bool $bAuth авторизовать в случае успешной регистрации
     * @return array|bool
     *  false - ошибка регистрации
     *  array - данные о вновь созданном пользователе (user_id, password, activate_link)
     */
    public function userRegister(array $aData, $bAuth = false)
    {
        # генерируем логин на основе email-адреса
        if (isset($aData['email'])) {
            $login = mb_substr($aData['email'], 0, mb_strpos($aData['email'], '@'));
            $login = preg_replace('/[^a-z0-9\_]/ui', '', $login);
            $login = mb_strtolower(trim($login, '_ '));
            if (mb_strlen($login) >= $this->loginMinLength) {
                if (mb_strlen($login) > $this->loginMaxLength) {
                    $login = mb_substr($login, 0, $this->loginMaxLength);
                }
                $aData['login'] = $this->model->userLoginGenerate($login, true);
            } else {
                $aData['login'] = $this->model->userLoginGenerate();
            }
        }

        # генерируем пароль или используем переданный
        $sPassword = (isset($aData['password']) ? $aData['password'] : func::generator(10));

        # подготавливаем данные
        $this->cleanUserData($aData);
        $aData['password_salt'] = $this->security->generatePasswordSalt();
        $aData['password'] = $this->security->getUserPasswordMD5($sPassword, $aData['password_salt']);

        # данные необходимые для активации аккаунта
        $this->getActivationInfo($sActivateCode, $sActivateLink, $sActivateExpire);
        $aData['activated'] = 0;
        $aData['activate_key'] = $sActivateCode;
        $aData['activate_expire'] = $sActivateExpire;

        # по-умолчанию подписываем на все типы email-уведомлений
        $aData['enotify'] = $this->getEnotifyTypes(0, true);

        # создаем аккаунт
        $nUserID = $this->model->userCreate($aData, self::GROUPID_MEMBER);
        if (!$nUserID) {
            return false;
        }

        if ($bAuth) {
            # авторизуем
            $this->userAuth($nUserID, 'user_id', $aData['password']);
        }

        return array(
            'user_id'       => $nUserID,
            'password'      => $sPassword,
            'activate_link' => $sActivateLink,
        );
    }

    function getActivationInfo(&$sCode, &$sLink, &$sExpire)
    {
        if (empty($sCode))
            $sCode = md5( substr( md5(uniqid(mt_rand().'*&^$',true)),0,10 ).time() );

        $sLink = static::url('activate', array('c'=>$sCode));

        $sExpire = date('Y-m-d H:i:s', strtotime('+'.config::sys('users.notactive.delete.timeout', 3).' days'));
    }
    
    function setPreAtivationFlag($nUserID, $mHash, $bCheck = false)
    {
        $sKey = 'users-reg-confirm-'.$nUserID;
        if($bCheck) {
            return (!empty($_SESSION[$sKey]) && $_SESSION[$sKey] == $mHash);
        } elseif($mHash === -1) {
            if(!empty($_SESSION[$sKey])) {
                unset($_SESSION[$sKey]);
            }
        } else {
            $_SESSION[$sKey] = $mHash;
        }
    }

    /**
     * @param $mHash
     * @param $aData
     * @param bool $bGet
     * @return mixed
     */
    function setPreRegisterSocialFlag($mHash, $aData, $bGet = false)
    {
        $sKey = 'users-prereg-social-'.$mHash;
        if($bGet) {
            return (!empty($_SESSION[$sKey]) ? $_SESSION[$sKey] : false);
        } elseif($aData === -1) {
            if(!empty($_SESSION[$sKey])) {
                unset($_SESSION[$sKey]);
            }
        } else {
            $_SESSION[$sKey] = $aData;
        }
    } 
    
    function prepareAjaxPreActivation(&$aResult)
    {
        $aData = config::get('__users_preactivate_data', array());
        $this->security->sessionStart(); // стартуем сессию для возможности хранения состояния
        // сохраняем последовательность подтверждения пароля
        $aData['hash'] = func::generator(10);
        $this->setPreAtivationFlag($aData['id'], $aData['hash'], false);
        $aResult['confirm_popup'] = $this->viewPHP($aData, 'popup.activation');
        return $aData;
    }
    
    function setRegisterSecondEnotifyFlag($nUserID, $bGet = false)
    {
        $this->security->sessionStart();

        $sKey = 'users-reg-email2';
        if($bGet) {
            return (!empty($_SESSION[$sKey]) ? $_SESSION[$sKey] : false);
        } elseif($nUserID === -1) {
            if(!empty($_SESSION[$sKey])) {
                unset($_SESSION[$sKey]);
            }
        } else {
            $_SESSION[$sKey] = $nUserID;
        }
    }
    
    function getEnotifyTypes($nSettings = 0, $bAllCheckedSettings = false)
    {
        $aTypes = array(
            USERS_ENOTIFY_NEWS => array( 'title' => _t('','Подписаться на рассылку новостей'), 'a' => 0 ),
        );
        
        if($bAllCheckedSettings) {
            return (!empty($aTypes) ? array_sum( array_keys($aTypes) ) : 0 );
        }
        
        if(!empty($nSettings)) {
            foreach($aTypes as $k=>$v) {
                if($nSettings & $k) {
                    $aTypes[$k]['a'] = 1;
                }
            }
        }
        
        return $aTypes;
    }

    /**
     * Формирование ключа для авторизации из под пользователя
     * @param integer $userID ID пользователя
     * @param string $userEmail E-mail пользователя
     * @return string
     */
    function frontendAuthHash($userID, $userEmail)
    {
        return $this->security->getRememberMePasswordMD5($userID.$userEmail);
    }

    /**
     * Иницилизация компонента работы с форумами
     * @return UsersForum
     */
    function forum()
    {
        static $i;
        if( ! isset($i)) {
            require_once $this->module_dir.'users.forum.php';
            $i = new UsersForum();
            $i->init();
        }
        return $i;
    }
    
}