<?php

class RealtyModel extends Model
{
    /** @var RealtyBase */
    var $controller;

    var $langCategories = array(
        'title'             => TYPE_NOTAGS, # Название
        'titleh1'           => TYPE_NOTAGS, # Заголовок h1
        'mtitle'            => TYPE_NOTAGS, # Meta Title
        'mkeywords'         => TYPE_NOTAGS, # Meta Keywords
        'mdescription'      => TYPE_NOTAGS, # Meta Description
        'tpl_descshort'     => TYPE_STR,    # шаблон для краткого описания
        'tpl_descfull'      => TYPE_STR,    # шаблон для подробного описания
    );

    var $langCategoriesTypes = array(
        'view_title'        => TYPE_STR,
        'short_title'       => TYPE_STR,
    );

    var $langTypes = array(
        'title'        => TYPE_STR, # название
    );

    /**
     * Получение данных объявления
     * @param integer $nItemID ID объявления
     * @param array|string $aFields требуемые поля объявления
     * @param array|string $aUserFields требуемые данные о пользователе (авторе объявления)
     * @return array
     */
    function itemData($nItemID, $aFields = '*', $aUserFields = array())
    {
        if( empty($nItemID) ) return array();

        $aParams = array();
        // item params
        if (empty($aFields)) $aFields = '*';
        if(!is_array($aFields)) $aFields = array($aFields);
        foreach($aFields as $v) {
            $aParams[] = 'I.'.$v;
        }
        // user params
        $bJoinUsers = false;
        if( ! empty($aUserFields) ) {
            if( is_string($aUserFields) ) $aUserFields = array($aUserFields);
            foreach($aUserFields as $v) {
                $aParams[] = 'U.'.$v;
            }
            $bJoinUsers = true;
        }

        return $this->db->one_array('SELECT '.join(',', $aParams).'
                                  FROM '.TABLE_REALTY_ITEMS.' I
                                       '.($bJoinUsers ? ', '.TABLE_USERS.' U' : '').'
                                  WHERE I.id = :id '.($bJoinUsers ? 'AND I.user_id = U.user_id' : ''),
                                  array(':id'=>$nItemID));
    }

    /**
     * Сохранение объявления
     * @param integer $nItemID ID объявления
     * @param array $aData данные объявления
     * @param mixed $sDynpropsDataKey ключ данных дин. свойств, например 'd' или FALSE
     * @return boolean|integer
     */
    function itemSave($nItemID, array $aData, $sDynpropsDataKey = false)
    {
        if (empty($aData)) return false;

        if ( ! empty($sDynpropsDataKey) && ! empty($aData['cat_id'])) {
            $aDataDP = $this->controller->dpSave($aData['cat_id'], $aData, $sDynpropsDataKey);
            $aData = array_merge($aData, $aDataDP);
        }

        $aData['modified'] = $this->db->now();

        if (isset($aData['filename']))
        {
            reset($aData['filename']);
            $aData['imgfav'] = current($aData['filename']);
            $aData['img'] = join(',', $aData['filename']);
            $aData['imgcnt'] = sizeof($aData['filename']);
            unset($aData['filename']);
        }

        if (isset($aData['price_params'])) {
            $aData['price_params'] = array_sum($aData['price_params']);
        }

        if (isset($aData['title'])) {
            $aData['title_alt'] = HTML::escape(strip_tags($aData['title']));
            $aData['keyword'] = mb_strtolower(func::translit($aData['title_alt']));
        }
        if ($nItemID > 0)
        {
            $res = $this->db->update(TABLE_REALTY_ITEMS, $aData, array('id'=>$nItemID));
        } else {
            $aData['created'] = $this->db->now();
            $res = $nItemID = $this->db->insert(TABLE_REALTY_ITEMS, $aData);
        }

        if ( ! $res) return $res;

        # формируем URL просмотра
        $aData = $this->itemData($nItemID, array('keyword', 'city_id'));
        $this->db->update(TABLE_REALTY_ITEMS, array(
            'link' => Realty::urlView($nItemID, $aData['keyword'], $aData['city_id']),
        ), array('id'=>$nItemID));

        return $res;
    }

    /**
     * Список объявлений пользователя в профиле (фронтенд)
     * @param integer $nUserID ID пользователя
     * @param bool $bCount только подсчет кол-ва объявлений
     * @param string $sqlLimit
     * @return array
     */
    function itemsListProfile($nUserID, $bCount = false, $sqlLimit = '') // frontend
    {
        $aFilter = $this->prepareFilter(array('user_id'=>$nUserID), 'I');
        if ($bCount) {
            return $this->db->one_data('SELECT COUNT(I.id)
                FROM '.TABLE_REALTY_ITEMS.' I '.$aFilter['where'], $aFilter['bind']);
        }

        return $this->db->select('SELECT I.id, I.link, I.status, I.status_prev,
                                I.moderated, I.blocked_num, I.blocked_reason, I.publicated_to,
                                I.svc, (I.svc & '.Realty::SERVICE_FIX.') as fixed, (I.svc & '.Realty::SERVICE_MARK.') as marked,
                                I.imgfav, I.address, I.descshort, I.price, I.price_curr, I.price_params as pricep
                            FROM '.TABLE_REALTY_ITEMS.' I
                            '.$aFilter['where'].'
                            ORDER BY I.id DESC
                        '.$sqlLimit, $aFilter['bind']);
    }

    /**
     * Список объявлений кампании (фронтенд)
     * @param integer $nCompanyID ID кампании
     * @param bool $bCount только подсчет кол-ва объявлений
     * @param string $sqlLimit
     * @return array
     */
    function itemsListCompany($nCompanyID, $bCount = false, $sqlLimit = '') // frontend
    {
        $aFilter = array(
            'company_id'=>$nCompanyID,
            'status'=>Realty::STATUS_PUBLICATED);

        if (Realty::premoderation()) {
            $aFilter[':moderated'] = 'I.moderated > 0';
        }

        $aFilter = $this->prepareFilter($aFilter, 'I');

        if($bCount) {
            return $this->db->one_data('SELECT COUNT(I.id) FROM '.TABLE_REALTY_ITEMS.' I '.$aFilter['where'], $aFilter['bind']);
        }

        return $this->db->select('SELECT I.id, I.link, I.status, I.status_prev,
                                I.moderated, I.blocked_num, I.blocked_reason, I.publicated_to,
                                I.svc, (I.svc & '.Realty::SERVICE_FIX.') as fixed, (I.svc & '.Realty::SERVICE_MARK.') as marked,
                                I.imgfav, I.address, I.descshort, I.price, I.price_curr, I.price_params as pricep
                                FROM '.TABLE_REALTY_ITEMS.' I
                                '.$aFilter['where'].'
                                ORDER BY fixed DESC, I.fixed_order DESC, I.publicated_order DESC
                            '.$sqlLimit, $aFilter['bind']);
    }

    function itemsListVIP($nCatID = 0, $nTypeID = 0, $nCityID = 0)
    {
        $isServiceOn = $this->controller->svcIsVIPEnabled($aVipSettings);
        if( ! $isServiceOn ) return array();

        $sqlFilter = array('(I.svc & '.Realty::SERVICE_VIP.')', 'I.vip_to >= :now', 'I.status = '.Realty::STATUS_PUBLICATED);
        if (true) {
            if( $nCatID > 0 ) { # категория: Квартира, Комната
                $sqlFilter[] = 'I.cat_id = '.$nCatID;
            }
            if($nTypeID > 0) { # тип: Продажа, Аренда
                $sqlFilter[] = 'I.type_id = '.$nTypeID;
            }
            if($nCityID > 0) { # регион: город
                $sqlFilter[] = 'I.city_id = '.$nCityID;
            }
        }

        $aData = $this->db->select('SELECT I.id, I.link, I.imgfav, I.title, I.title_alt, I.price, I.price_curr, I.price_params as pricep
                FROM '.TABLE_REALTY_ITEMS.' I
                WHERE '.join(' AND ', $sqlFilter).'
                ORDER BY RAND()
                '. $this->db->prepareLimit(0, $aVipSettings['cnt']),
                array(':now'=>$this->db->now()));
        if( empty($aData) ) $aData = array();
        return $aData;
    }

    /**
     * Перестраивание ссылок во всех записях
     */
    public function itemsLinksRebuild()
    {
        $model = $this;
        $this->db->select_rows_chunked(TABLE_REALTY_ITEMS, array('id','keyword','city_id'),
                                       array(),'id',array(), function ($items) use ($model) {
            foreach ($items as $v) {
                $model->db->update(TABLE_REALTY_ITEMS, array(
                    'link' => Realty::urlView($v['id'], $v['keyword'], $v['city_id'])
                ), array('id'=>$v['id']));
            }
        }, 100);
    }

    /**
     * Сохранение категории
     * @param integer $nCategoryID ID категории
     * @param array $aDataAll данные категории
     * @return boolean|integer
     */
    function categorySave($nCategoryID, $aDataAll)
    {
        if (empty($aDataAll)) return false;

        $aData = array_diff_key($aDataAll, $this->langCategories);
        $aData['title'] = $aDataAll['title'][LNG];

        if($nCategoryID > 0)
        {
            $aData['modified'] = $this->db->now(); # Дата изменения

            $this->categoryTypesSave($nCategoryID, $aData);

            $res = $this->db->update(TABLE_REALTY_CATEGORIES, $aData, array('id'=>$nCategoryID));
            $this->db->langUpdate($nCategoryID, $aDataAll, $this->langCategories, TABLE_REALTY_CATEGORIES_LANG);
            return !empty($res);
        } else {
            $aData['created'] = $this->db->now(); # Дата создания
            $aData['modified'] = $this->db->now(); # Дата изменения

            $this->categoryTypesSave(0, $aData);
            $nCategoryID = $this->db->insert(TABLE_REALTY_CATEGORIES, $aData);
            if( $nCategoryID > 0 ) {
                $this->db->langInsert($nCategoryID, $aDataAll, $this->langCategories, TABLE_REALTY_CATEGORIES_LANG);
                $this->categoryTypesSave($nCategoryID);
            }
            return $nCategoryID;
        }
    }

    /**
     * Сохранение типов категории
     * @param integer $nCategoryID ID категории
     * @param array $aData @ref данные категории
     */
    private function categoryTypesSave($nCategoryID = 0, & $aData = array())
    {
        static $aTypes = array();

        if( ! empty($aData)){
            if( ! empty($aData['types'])){
                $aTypes = $aData['types'];
            }
            unset($aData['types']);
        }
        if( ! empty($aTypes) && $nCategoryID){
            $aItemsCount = $this->db->select_key('
                SELECT type_id, COUNT(id) as items
                FROM '.TABLE_REALTY_ITEMS.'
                WHERE cat_id = :cat
                GROUP BY type_id', 'type_id', array(':cat' => $nCategoryID));

            $this->db->delete(TABLE_REALTY_CATEGORIES_TYPES, array('cat_id'=>$nCategoryID));

            foreach($aTypes as $t){
                if (empty($t['type_id'])) continue;
                $nTypeID = $t['type_id'];
                $t['cat_id']  = $nCategoryID;
                if( ! empty($aItemsCount[ $nTypeID ]['items'])){
                    $t['items'] = $aItemsCount[ $nTypeID ]['items'];
                }
                $this->db->langFieldsModify($t, $this->langCategoriesTypes, $t);
                $this->db->insert(TABLE_REALTY_CATEGORIES_TYPES, $t);
            }
        }
    }

    function categoryDataSearch($nCategoryID)
    {
        return $this->db->one_array('SELECT C.id, C.keyword,
                CL.title, CL.titleh1, CL.mtitle, CL.mkeywords, CL.mdescription, C.mtemplate
            FROM '.TABLE_REALTY_CATEGORIES.' C,
                 '.TABLE_REALTY_CATEGORIES_LANG.' CL
            WHERE C.id = :id'.$this->db->langAnd(true, 'C', 'CL'),
            array(':id' => $nCategoryID));
    }

    function typeDataSearch($nTypeID)
    {
        return $this->db->one_array('SELECT id, title_'.LNG.' as title, keyword
            FROM '.TABLE_REALTY_TYPES.' WHERE id = :id', array(':id' => $nTypeID));
    }

    function categoryIDByKeyword($sKeyword)
    {
        return (int)$this->db->select_data(TABLE_REALTY_CATEGORIES, 'id', array('keyword'=>$sKeyword));
    }

    function typeIDByKeyword($sKeyword)
    {
        return (int)$this->db->select_data(TABLE_REALTY_TYPES, 'id', array('keyword'=>$sKeyword));
    }

    function getLocaleTables()
    {
        return array(
            TABLE_REALTY_CATEGORIES => array('type'=>'table','fields'=>$this->langCategories),
            TABLE_REALTY_CATEGORIES_TYPES => array('type'=>'fields','fields'=>$this->langCategoriesTypes),
            TABLE_REALTY_TYPES => array('type'=>'fields','fields'=>$this->langTypes),
            TABLE_REALTY_DYNPROPS => array('type'=>'fields','fields'=>array('title'=>TYPE_NOTAGS)),
            TABLE_REALTY_DYNPROPS_MULTI => array('type'=>'fields','fields'=>array('name'=>TYPE_NOTAGS)),
        );
    }

}